import os
import datetime
import time

from library.python import resource
from jinja2 import BaseLoader, TemplateNotFound, Environment, FileSystemLoader


def is_inside_yatest():
    try:
        import yatest
    except ImportError:
        return False

    try:
        yatest.common.runtime._get_ya_config()
    except NotImplementedError:
        return False

    return True


def to_unicode(content):
    try:
        return content.decode("utf-8")
    except UnicodeEncodeError:
        return content


class ResourceLoader(BaseLoader):

    """ Allow render jinja templates from Resources """

    def __init__(self, prefixes):
        self.prefixes = prefixes

    def get_source(self, environment, template):
        for prefix in self.prefixes:
            path = os.path.join(prefix, template)
            source = resource.find(path)
            if source:
                break
        else:
            path = template
            source = resource.find(path)
        if source:
            return to_unicode(source), path, lambda: True
        raise TemplateNotFound("Template {0!r}, paths: {1!r}".format(template, self.prefixes))


class JinjaMixin(object):

    """ Render from string """

    def render(self, template, **kwargs):
        return Environment().from_string(template).render(**self.get_context_data(**kwargs))

    def get_context_data(self, **kwargs):
        context = kwargs
        # context.update(os.environ) # do not pass environment to template anymore
        # allow use python datetime in templates {{ datetime. ... }}
        context.update(datetime=datetime, time=time, is_test_run=is_inside_yatest())
        return context


class JinjaFileSystemMixin(JinjaMixin):

    """ Render from filesystem (allow include in templates) """

    def render(self, template, **kwargs):
        base_path, template_name = os.path.split(template)
        template = Environment(loader=FileSystemLoader((base_path, os.path.dirname(base_path)))).get_template(
            template_name
        )
        return template.render(**self.get_context_data(**kwargs))


class JinjaResourceMixin(JinjaMixin):

    """ Render from resource (allow include in templates) """

    @property
    def _jinja_extensions(self):
        return []

    def render(self, template, **kwargs):
        base_path, template_name = os.path.split(template)
        template = Environment(
            loader=ResourceLoader((base_path, os.path.dirname(base_path))), extensions=self._jinja_extensions
        ).get_template(template_name)
        return template.render(**self.get_context_data(**kwargs))


def filesystem_render(template, **context):
    return JinjaFileSystemMixin().render(template, **context)


def resource_render(template, **context):
    return JinjaResourceMixin().render(template, **context)
