import re
import requests
from retry import retry


class JugglerError(Exception):
    pass


def report_event_to_juggler(status, service, host, logger, description='', tags=None, tries=3, delay=5,
                            juggler_api_url='http://juggler-push.search.yandex.net/events'):
    """For more info https://wiki.yandex-team.ru/sm/juggler/batching/"""
    service = re.sub("[^0-9a-zA-Z\\+_\\-\\./]+", "+", service)

    @retry(JugglerError, tries=tries, delay=delay, logger=logger)
    def send_event(url, event):
        response = requests.post(url, json=event)
        info_message = '[{}] {}'.format(response.status_code, response.content)
        logger.info(info_message)

        if response.status_code >= 500:
            raise JugglerError(info_message)

    event = {
        'events': [
            {
                'host': host,
                'service': service,
                'description': description,
                'status': status,
                'instance': '',
                'tags': tags or [],
            }
        ]
    }

    logger.info('Sending event to juggler: {}'.format(event))
    try:
        send_event(juggler_api_url, event)
    except Exception as error:
        logger.warning('Sending event to Juggler failed: {}'.format(error.message))
