import json
import logging
import subprocess


logger = logging.getLogger(__name__)


class JugglerQueue:
    def __init__(self, dry_run):
        self.dry_run = dry_run

    def enqueue_ok(self, host, service, desc):
        self.enqueue_event(0, host, service, desc)

    def enqueue_warn(self, host, service, desc):
        self.enqueue_event(1, host, service, desc)

    def enqueue_crit(self, host, service, desc):
        self.enqueue_event(2, host, service, desc)

    def enqueue_event(self, status, host, service, desc):
        command = ["juggler_queue_event", "--host={}".format(host), "-n", service, "-s", str(status), "-d", desc]
        logger.debug(command)

        if not self.dry_run:
            subprocess.call(command)


class JugglerBatchingContext(object):
    def __init__(self, dry_run):
        self.dry_run = dry_run

    def __enter__(self):
        self.messages = []
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        command = ["sudo", "juggler_queue_event", "--batch"]
        serialized_messages = json.dumps(self.messages)
        logger.debug(serialized_messages)

        if not self.dry_run:
            pipe = subprocess.Popen(command, stdin=subprocess.PIPE)
            pipe.communicate(serialized_messages)

    def enqueue_ok(self, host, service, desc):
        self.enqueue_event("OK", host, service, desc)

    def enqueue_warn(self, host, service, desc):
        self.enqueue_event("WARN", host, service, desc)

    def enqueue_crit(self, host, service, desc):
        self.enqueue_event("CRIT", host, service, desc)

    def enqueue_event(self, status, host, service, desc):
        self.messages.append({
            "status": status,
            "host": host,
            "service": service,
            "description": desc
        })
