import logging
import multiprocessing
from six.moves import queue
import sys
import threading

from library.python.deploy_formatter import DeployFormatter


def configure_stream_logger(logger, stream, level=logging.INFO, formatter=None, filter_=None):
    logger.setLevel(level)
    handler = logging.StreamHandler(stream=stream)
    formatter = formatter or logging.Formatter("[%(asctime)s] [%(levelname)s] %(message)s")
    handler.setFormatter(formatter)

    if filter_ is not None:
        handler.addFilter(filter_)

    logger.addHandler(handler)


def configure_stdout_logger(logger, level=logging.INFO):
    configure_stream_logger(logger, sys.stdout, level)


def configure_deploy_logger(logger, level=logging.INFO):
    configure_stream_logger(logger, sys.stdout, level, DeployFormatter())


def configure_stderr_logger(logger, level=logging.INFO):
    configure_stream_logger(logger, sys.stderr, level)


def register_stream_logger(logger_name, stream, level=logging.INFO):
    logger = logging.getLogger(logger_name)
    configure_stream_logger(logger, stream, level)
    return logger


def register_stdout_logger(logger_name, level=logging.INFO):
    return register_stream_logger(logger_name, sys.stdout, level)


def register_stderr_logger(logger_name, level=logging.INFO):
    return register_stream_logger(logger_name, sys.stderr, level)


# https://github.com/jruere/multiprocessing-logging/blob/master/multiprocessing_logging.py
class MultiProcessingHandler(logging.Handler):
    def __init__(self, sub_handler):
        super(MultiProcessingHandler, self).__init__()

        self.sub_handler = sub_handler

        self.setLevel(self.sub_handler.level)
        self.setFormatter(self.sub_handler.formatter)
        self.filters = self.sub_handler.filters

        self.queue = multiprocessing.Queue()
        self._is_closed = False

        self._receive_thread = threading.Thread(target=self._receive)
        self._receive_thread.daemon = True
        self._receive_thread.start()

    def _receive(self):
        while True:
            try:
                if self._is_closed and self.queue.empty():
                    break

                record = self.queue.get(timeout=1)
                self.sub_handler.emit(record)
            except EOFError:
                break
            except queue.Empty:
                pass

        self.queue.close()
        self.queue.join_thread()

    def _format_record(self, record):
        # ensure that exc_info and args have been stringified. Removes any chance of unpickleable things inside
        if record.args:
            record.msg = record.msg % record.args
            record.args = None
        if record.exc_info:
            self.format(record)
            record.exc_info = None

        return record

    def emit(self, record):
        try:
            msg = self._format_record(record)
            self.queue.put_nowait(msg)
        except Exception:
            self.handleError(record)

    def close(self):
        if not self._is_closed:
            self._is_closed = True
            self._receive_thread.join(5.0)

            self.sub_handler.close()
            super(MultiProcessingHandler, self).close()


def convert_handlers_to_mp(logger):
    for orig_handler in list(logger.handlers):
        handler = MultiProcessingHandler(sub_handler=orig_handler)

        logger.removeHandler(orig_handler)
        logger.addHandler(handler)


def configure_multiprocessing_stdout_logger(logger, level=logging.INFO):
    configure_stream_logger(logger, sys.stdout, level, logging.Formatter("[%(asctime)s] [%(levelname)s] [%(process)d] %(message)s"))
    convert_handlers_to_mp(logger)


def configure_multiprocessing_deploy_logger(logger, level=logging.INFO, filter_=None):
    configure_stream_logger(logger, sys.stdout, level, DeployFormatter(), filter_=filter_)
    convert_handlers_to_mp(logger)


def get_function_logger(logger):
    def function_logger(function):
        def wrapper(*args, **kwargs):
            logger.info('Function "{}" is started'.format(function.__name__))
            function_res = function(*args, **kwargs)
            logger.info('Function "{}" is successfully finished'.format(function.__name__))
            return function_res
        return wrapper
    return function_logger
