#pragma once

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/operation.h>
#include <library/cpp/any/any.h>

#include <type_traits>
#include <util/system/type_name.h>
#include <util/generic/buffer.h>

namespace NNativeYT {
    bool IsRegistered(const TString& name);

    struct TJobResult {
        const NYT::TOperationId Operation;
    };

    using TPaths = TVector<NYT::TRichYPath>;

    struct TOperationArgs {
        NAny::TAny Spec;
        TPaths Source;
        TPaths Destination;
        NYT::ITransactionPtr Transaction;
        TString Proxy;
        struct {
            TBuffer Mapper;
            TBuffer Reducer;
            TBuffer Combiner;
        } State;
        struct {
            TVector<NYT::TRichYPath> Mapper;
            TVector<NYT::TRichYPath> Reducer;
        } Files;
        NYT::TNode OptionsSpec;
    };
    using TJob = std::function<TJobResult(const TOperationArgs& args)>;

    class TJobRegistrar {
    public:
        static TJobRegistrar* Get();
        TJob Job(const TString& name) {
            if (!Jobs.contains(name)) {
                ythrow yexception() << "No such job " << name;
            }
            return Jobs.at(name);
        }
        bool Has(const TString& name) const {
            return Jobs.contains(name);
        }
        void Register(const TString& name, const TJob& job) {
            if (Jobs.contains(name)) {
                ythrow yexception() << "Can't register " << name << " twice";
            }
            Jobs[name] = job;
        }

    private:
        THashMap<TString, TJob> Jobs;
    };

    class TRegistrator {
    public:
        TRegistrator(const TString& name, const std::function<void()> rightNow, const TJob& job) {
            rightNow();
            TJobRegistrar::Get()->Register(name, job);
        }
    };

    NYT::TOperationId Execute(const TString& name,
                              const NAny::TAny& spec,
                              const TOperationArgs& args);

    template <class... Args>
    TString CombinedName(const Args... args) {
        std::vector<TString> vec = {args...};
        auto it = vec.begin();
        TString sep = "|";
        TString result = *it;
        while (++it != vec.end()) {
            result += sep + (*it);
        }
        return result;
    }

    template <class TOperationSpec, class TReaderRowType>
    void AddSameInputPaths(TOperationSpec& spec, const TPaths& sources) {
        for (const auto& each : sources) {
            spec.template AddInput<TReaderRowType>(each);
        }
    }

    template <class TOperationSpec, class TWriterRowType>
    void AddSameOutputPaths(TOperationSpec& spec, const TPaths& destinations) {
        for (const auto& each : destinations) {
            spec.template AddOutput<TWriterRowType>(each);
        }
    }

    template <class TOperationSpec, typename TRowTypes, size_t... each>
    void AddProtoInputPaths(TOperationSpec& spec, const TPaths& sources, std::integer_sequence<size_t, each...> seq) {
        Y_ENSURE(seq.size() == sources.size(), "Invalid number of inputs");
        ((spec.template AddInput<typename std::tuple_element<each, TRowTypes>::type>(sources.at(each))), ...);
    }

    template <class TOperationSpec, typename TRowTypes, size_t... each>
    void AddProtoOutputPaths(TOperationSpec& spec, const TPaths& destinations, std::integer_sequence<size_t, each...> seq) {
        Y_ENSURE(seq.size() == destinations.size(), "Invalid number of outputs");
        ((spec.template AddOutput<typename std::tuple_element<each, TRowTypes>::type>(destinations.at(each))), ...);
    }

    template<typename TInputOp, typename TOutputOp = TInputOp>
    struct TAddPathsHelper {
        template<class TOperationSpec>
        void operator()(TOperationSpec& spec, const TPaths& sources, const TPaths& destinations) {
            if constexpr (std::is_same<typename TInputOp::TReader::TRowType, ::google::protobuf::Message>::value) {
                AddProtoInputPaths<TOperationSpec, typename TInputOp::TInputs>(spec, sources, std::make_index_sequence<std::tuple_size<typename TInputOp::TInputs>::value>{});
            } else {
                AddSameInputPaths<TOperationSpec, typename TInputOp::TReader::TRowType>(spec, sources);
            }

            if constexpr (std::is_same<typename TOutputOp::TWriter::TRowType, ::google::protobuf::Message>::value) {
                AddProtoOutputPaths<TOperationSpec, typename TOutputOp::TOutputs>(spec, destinations, std::make_index_sequence<std::tuple_size<typename TOutputOp::TOutputs>::value>{});
            } else {
                AddSameOutputPaths<TOperationSpec, typename TOutputOp::TWriter::TRowType>(spec, destinations);
            }
        }
    };

    template <class TArgsSpec>
    void InitOptions(NYT::TOperationOptions& options, const TArgsSpec& spec) {
        options.JobCommandPrefix("Y_PYTHON_ENTRY_POINT=\"crypta.lib.python.native_yt.entry.point\" ");
        options.Wait(false);
        options.Spec(spec);
    }

    template <class TOperation, bool>
    struct TCreateWithState {
    };

    template <class TOperation>
    struct TCreateWithState<TOperation, true> {
        TOperation* operator()(const TBuffer& state) const {
            return new TOperation(state);
        }
    };

    template <class TOperation>
    struct TCreateWithState<TOperation, false> {
        TOperation* operator()(const TBuffer& state) const {
            if (!state.Empty()) {
                ythrow yexception() << TypeName<TOperation>() << " is stateless but state is provided";
            }
            return new TOperation();
        }
    };

    template <class TOperation>
    TOperation* Create(const TBuffer& state) {
        return TCreateWithState<TOperation, std::is_constructible<TOperation, TBuffer>::value>()(state);
    }

    template <class TSpec>
    TSpec GetSpec(const TOperationArgs& args) {
        const auto& spec = args.Spec;
        if (spec.Compatible<TSpec>()) {
            return spec.Cast<TSpec>();
        } else {
            ythrow yexception() << "Can not obtain " << TypeName<TSpec>()
                                << "the operation ought to be registered wrong way";
        }
    }

}

#define REGISTER_REDUCER_(NAME)                      \
    if (!NNativeYT::IsRegistered(ToString(#NAME))) { \
        REGISTER_REDUCER(NAME);                      \
    }

#define REGISTER_MAPPER_(NAME)                       \
    if (!NNativeYT::IsRegistered(ToString(#NAME))) { \
        REGISTER_MAPPER(NAME);                       \
    }

#define CYT_REGISTER_MAPPER_(NAME, ADD_PATHS)                                           \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                  \
        #NAME,                                                                          \
        []() -> decltype(auto) {                                                        \
            REGISTER_MAPPER_(NAME);                                                     \
        },                                                                              \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {            \
            using TSpec = NYT::TMapOperationSpec;                                       \
            auto realSpec = NNativeYT::GetSpec<TSpec>(args);                            \
            auto options = NYT::TOperationOptions();                                    \
            ADD_PATHS{}(realSpec, args.Source, args.Destination);                       \
            NNativeYT::InitOptions(options, args.OptionsSpec);                          \
            auto mapper = NNativeYT::Create<NAME>(args.State.Mapper);                   \
            auto operation = args.Transaction->Map(realSpec, mapper, options);          \
            return {.Operation = operation->GetId()};                                   \
        });

#define CYT_TRY_REDUCE_(NAME, ADD_PATHS, SPEC, OP)                                 \
    if (args.Spec.Compatible<SPEC>()) {                                            \
        auto realSpec = NNativeYT::GetSpec<SPEC>(args);                            \
        auto options = NYT::TOperationOptions();                                   \
        ADD_PATHS{}(realSpec, args.Source, args.Destination);                      \
        NNativeYT::InitOptions(options, args.OptionsSpec);                         \
        auto reducer = NNativeYT::Create<NAME>(args.State.Reducer);                \
        auto operation = args.Transaction->OP(realSpec, reducer, options);         \
        return {.Operation = operation->GetId()};                                  \
    }

#define CYT_REGISTER_REDUCER_(NAME, ADD_PATHS)                                          \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                  \
        #NAME,                                                                          \
        []() -> decltype(auto) {                                                        \
            REGISTER_REDUCER_(NAME)                                                     \
        },                                                                              \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {            \
            CYT_TRY_REDUCE_(NAME, ADD_PATHS, NYT::TReduceOperationSpec, Reduce)         \
            CYT_TRY_REDUCE_(NAME, ADD_PATHS, NYT::TJoinReduceOperationSpec, JoinReduce) \
            ythrow yexception() << #NAME << " is neither reducer, nor join-reducer";    \
        });

#define CYT_REGISTER_MAPREDUCER_(MAPPER_NAME, REDUCER_NAME, ADD_PATHS)                        \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                        \
        NNativeYT::CombinedName(#MAPPER_NAME, #REDUCER_NAME),                                 \
        []() -> decltype(auto) {                                                              \
            REGISTER_MAPPER_(MAPPER_NAME);                                                    \
            REGISTER_REDUCER_(REDUCER_NAME);                                                  \
        },                                                                                    \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {                  \
            using TSpec = NYT::TMapReduceOperationSpec;                                       \
            auto realSpec = NNativeYT::GetSpec<TSpec>(args);                                  \
            auto options = NYT::TOperationOptions();                                          \
            ADD_PATHS{}(realSpec, args.Source, args.Destination);                             \
            NNativeYT::InitOptions(options, args.OptionsSpec);                                \
            auto mapper = NNativeYT::Create<MAPPER_NAME>(args.State.Mapper);                  \
            auto reducer = NNativeYT::Create<REDUCER_NAME>(args.State.Reducer);               \
            auto operation = args.Transaction->MapReduce(realSpec, mapper, reducer, options); \
            return {.Operation = operation->GetId()};                                         \
        });

#define CYT_REGISTER_MAPREDUCER_WITH_NULL_MAPPER_(REDUCER_NAME, ADD_PATHS)                     \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                         \
        NNativeYT::CombinedName("", #REDUCER_NAME),                                            \
        []() -> decltype(auto) {                                                               \
            REGISTER_REDUCER_(REDUCER_NAME);                                                   \
        },                                                                                     \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {                   \
            using TSpec = NYT::TMapReduceOperationSpec;                                        \
            auto realSpec = NNativeYT::GetSpec<TSpec>(args);                                   \
            auto options = NYT::TOperationOptions();                                           \
            ADD_PATHS{}(realSpec, args.Source, args.Destination);                              \
            NNativeYT::InitOptions(options, args.OptionsSpec);                                 \
            auto reducer = NNativeYT::Create<REDUCER_NAME>(args.State.Reducer);                \
            auto operation = args.Transaction->MapReduce(realSpec, nullptr, reducer, options); \
            return {.Operation = operation->GetId()};                                          \
        });

#define CYT_REGISTER_MAPREDUCER_WITH_COMBINER_(MAPPER_NAME, COMBINER_NAME, REDUCER_NAME, ADD_PATHS)     \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                                  \
        NNativeYT::CombinedName(#MAPPER_NAME, #COMBINER_NAME, #REDUCER_NAME),                           \
        []() -> decltype(auto) {                                                                        \
            CYT_REGISTER_MAPREDUCER(MAPPER_NAME, REDUCER_NAME);                                         \
            REGISTER_REDUCER_(COMBINER_NAME);                                                           \
        },                                                                                              \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {                            \
            using TSpec = NYT::TMapReduceOperationSpec;                                                 \
            auto realSpec = NNativeYT::GetSpec<TSpec>(args);                                            \
            auto options = NYT::TOperationOptions();                                                    \
            ADD_PATHS{}(realSpec, args.Source, args.Destination);                                       \
            NNativeYT::InitOptions(options, args.OptionsSpec);                                          \
            auto mapper = NNativeYT::Create<MAPPER_NAME>(args.State.Mapper);                            \
            auto combiner = NNativeYT::Create<COMBINER_NAME>(args.State.Combiner);                      \
            auto reducer = NNativeYT::Create<REDUCER_NAME>(args.State.Reducer);                         \
            auto operation = args.Transaction->MapReduce(realSpec, mapper, combiner, reducer, options); \
            return {.Operation = operation->GetId()};                                                   \
        });

#define CYT_REGISTER_MAPREDUCER_WITH_COMBINER_WITH_NULL_MAPPER_(COMBINER_NAME, REDUCER_NAME, ADD_PATHS)  \
    static NNativeYT::TRegistrator Y_GENERATE_UNIQUE_ID(TRegistrator)(                                   \
        NNativeYT::CombinedName("", #COMBINER_NAME, #REDUCER_NAME),                                      \
        []() -> decltype(auto) {                                                                         \
            CYT_REGISTER_MAPREDUCER_WITH_NULL_MAPPER(REDUCER_NAME);                                      \
            REGISTER_REDUCER_(COMBINER_NAME);                                                            \
        },                                                                                               \
        [](const NNativeYT::TOperationArgs& args) -> NNativeYT::TJobResult {                             \
            using TSpec = NYT::TMapReduceOperationSpec;                                                  \
            auto realSpec = NNativeYT::GetSpec<TSpec>(args);                                             \
            auto options = NYT::TOperationOptions();                                                     \
            ADD_PATHS{}(realSpec, args.Source, args.Destination);                                        \
            NNativeYT::InitOptions(options, args.OptionsSpec);                                           \
            auto combiner = NNativeYT::Create<COMBINER_NAME>(args.State.Combiner);                       \
            auto reducer = NNativeYT::Create<REDUCER_NAME>(args.State.Reducer);                          \
            auto operation = args.Transaction->MapReduce(realSpec, nullptr, combiner, reducer, options); \
            return {.Operation = operation->GetId()};                                                    \
        });

#define CYT_REGISTER_MAPPER(NAME) CYT_REGISTER_MAPPER_(NAME, NNativeYT::TAddPathsHelper<NAME>);
#define CYT_REGISTER_REDUCER(NAME) CYT_REGISTER_REDUCER_(NAME, NNativeYT::TAddPathsHelper<NAME>);
#define CYT_REGISTER_MAPREDUCER(MAPPER_NAME, REDUCER_NAME) \
    CYT_REGISTER_MAPREDUCER_(MAPPER_NAME, REDUCER_NAME, (NNativeYT::TAddPathsHelper<MAPPER_NAME, REDUCER_NAME>)); \

#define CYT_REGISTER_MAPREDUCER_WITH_NULL_MAPPER(REDUCER_NAME) \
    CYT_REGISTER_MAPREDUCER_WITH_NULL_MAPPER_(REDUCER_NAME, NNativeYT::TAddPathsHelper<REDUCER_NAME>);
#define CYT_REGISTER_MAPREDUCER_WITH_COMBINER(MAPPER_NAME, COMBINER_NAME, REDUCER_NAME) \
    CYT_REGISTER_MAPREDUCER_WITH_COMBINER_(MAPPER_NAME, COMBINER_NAME, REDUCER_NAME, (NNativeYT::TAddPathsHelper<MAPPER_NAME, REDUCER_NAME>));

#define CYT_REGISTER_MAPREDUCER_WITH_COMBINER_WITH_NULL_MAPPER(COMBINER_NAME, REDUCER_NAME) \
    CYT_REGISTER_MAPREDUCER_WITH_COMBINER_WITH_NULL_MAPPER_(COMBINER_NAME, REDUCER_NAME, (NNativeYT::TAddPathsHelper<COMBINER_NAME, REDUCER_NAME>));
