import six

from cpython.exc cimport PyErr_CheckSignals
from util.generic.vector cimport TVector
from util.generic.string cimport TString, TStringBuf
from util.generic.hash cimport THashMap
from libcpp cimport bool
import os
import sys
import time
import logging

from yt import yson
import yt.wrapper as yt

logger = logging.getLogger(__name__)


cdef extern from "crypta/lib/python/native_yt/cpp/init.h" nogil:
    void Initialize(int argc, const char** argv)


cdef extern from "util/generic/buffer.h":
    cdef cppclass TBuffer nogil:
        TBuffer()
        TBuffer(const char* data, size_t length) except +


cdef extern from "util/generic/guid.h":
    cdef cppclass TGUID nogil:
        pass

    TString GetGuidAsString(const TGUID& guid) except +


cdef extern from *:
    ctypedef int i32
    ctypedef long long i64

cdef extern from "mapreduce/yt/interface/client.h" namespace "NYT":

    cdef cppclass ITransactionPtr:
        pass

    cdef cppclass TReadLimit:
        TReadLimit()
        TReadLimit& RowIndex(i64 index)

    cdef cppclass TReadRange:
        TReadRange()
        @staticmethod
        TReadRange FromRowIndices(i64 lowerLimit, i64 upperLimit)

        TReadRange& LowerLimit(const TReadLimit& readLimit)
        TReadRange& UpperLimit(const TReadLimit& readLimit)

    cdef cppclass TRichYPath:
        TRichYPath()
        TRichYPath(const char* path)
        TRichYPath& Foreign(const bool value)
        TRichYPath& AddRange(const TReadRange& range)
        TRichYPath& Append(const bool value)
        TRichYPath& Columns(const TVector[TString]& value)
        TRichYPath& RenameColumns(const THashMap[TString, TString]& value)

cdef extern from "mapreduce/yt/interface/node.h" namespace "NYT":
    cdef cppclass TNode:
        TNode()
        TNode(bool b)
        TNode(const char* s)
        TNode(const TString& s)
        TNode(unsigned long long ui)
        TNode(double d)
        TNode(const TNode& rhs)
        TNode(TNode&& rhs)

        void Add(const TString& s)

        TNode& operator ()(const TString& key, const TNode& value)

        @staticmethod
        TNode CreateList()

cdef extern from "mapreduce/yt/interface/common.h" namespace "NYT":
    cdef cppclass TTableSchema:
        pass

cdef extern from "mapreduce/yt/interface/serialize.h" namespace "NYT":
    void Deserialize(TRichYPath& path, const TNode& node)

cdef extern from "library/cpp/yson/node/node_io.h" namespace "NYT":
    TNode NodeFromYsonString(const TStringBuf input)

cdef extern from "crypta/lib/python/native_yt/cpp/client.h" namespace "NNativeYT":
    ITransactionPtr AttachTransaction(const TString& proxy, const TString& token, const TString& transaction) nogil except +


cdef extern from "crypta/lib/python/native_yt/cpp/operations.h" namespace "NNativeYT":
    cdef cppclass TOperationArgs nogil:
        TVector[TRichYPath] Source
        TVector[TRichYPath] Destination
        ITransactionPtr Transaction
        TString Proxy
        cppclass State nogil:
            TBuffer Mapper
            TBuffer Reducer
            TBuffer Combiner
        cppclass Files nogil:
            TVector[TRichYPath] Mapper;
            TVector[TRichYPath] Reducer;
        State State
        Files Files
        TNode OptionsSpec

    bool IsRegistered(const TString& name)
    TGUID RunMap(const TString& mapper_name, const TOperationArgs& args) nogil except +
    TGUID RunReduce(const TString& reducer_name, const TVector[TString] &sort_by, const TVector[TString] &reduce_by, const TOperationArgs& args) nogil except +
    TGUID RunMapReduce(const TString& mapper_name, const TString& reducer_name, const TVector[TString] &reduce_by, const TVector[TString] &sort_by, const TOperationArgs& args) nogil except +
    TGUID RunMapReduceWithCombiner(const TString& mapper_name, const TString& combiner_name, const TString& reducer_name, const TVector[TString] &reduce_by, const TVector[TString] &sort_by, const TOperationArgs& args) nogil except +
    TGUID RunJoinReduce(const TString& reducer_name, const TVector[TString] &join_by, const TOperationArgs& args) nogil except +
    bool IsOperationComplete(const ITransactionPtr& transaction, const TGUID& guid) nogil except +


cdef _to_binary_if_string_type(value):
    if isinstance(value, six.binary_type):
        # Simple return won't work for yson.YsonString which is instance of six.binary_type but not bytes
        return bytes(value)
    if isinstance(value, six.string_types):
        return six.ensure_binary(value)
    return value


cdef _transform_one_or_few_strings_to_list(one_or_few):
    return one_or_few if isinstance(one_or_few, (list, tuple)) else [one_or_few]


cdef TVector[TRichYPath] _paths(one_or_few) except *:
    cdef TVector[TRichYPath] result
    cdef TRichYPath path

    for each in _transform_one_or_few_strings_to_list(one_or_few):
        path = TRichYPath()
        Deserialize(path, NodeFromYsonString(yson.dumps(yt.TablePath(each).to_yson_type())))
        result.push_back(path)

    return result


cdef TVector[TString] list_to_TVector(one_or_few) except *:
    cdef TVector[TString] result
    for each in _transform_one_or_few_strings_to_list(one_or_few):
        result.push_back(_to_binary_if_string_type(each))
    return result

cdef TNode convert_list_to_tnode(_list) except *:
    cdef TNode lst = TNode.CreateList()
    for each in _list:
        lst.Add(_to_binary_if_string_type(each))
    return lst


cdef TNode convert_dict_to_tnode(_dict) except *:
    cdef TNode node
    cdef unsigned long long ull_value
    cdef bool bool_value
    cdef double double_value

    for key, value in _dict.items():
        key = _to_binary_if_string_type(key)
        if isinstance(value, dict):
            node(bytes(key), convert_dict_to_tnode(value))
        elif isinstance(value, list):
            node(bytes(key), convert_list_to_tnode(value))
        else:
            value = _to_binary_if_string_type(value)
            # isinstance(value, bool) does not work here due to name clash
            if isinstance(value, True.__class__):
                bool_value = value
                node(bytes(key), TNode(bool_value))
            elif isinstance(value, six.integer_types):
                ull_value = value
                node(bytes(key), TNode(ull_value))
            elif isinstance(value, float):
                double_value = value
                node(bytes(key), TNode(double_value))
            else:
                node(bytes(key), TNode(bytes(value)))
    return node


cdef TOperationArgs _prepare_args(
    source,
    destination,
    proxy,
    transaction,
    token,
    pool,
    title,
    script_name,
    mapper_state,
    reducer_state,
    dict spec,
    combiner_state,
    list mapper_files,
    list reducer_files
) except *:
    cdef bytes the_token = _to_binary_if_string_type(token or os.environ.get('YT_TOKEN'))
    cdef bytes the_proxy = _to_binary_if_string_type(proxy)
    cdef bytes the_transaction = _to_binary_if_string_type(transaction)

    assert the_token, ValueError('Token should be provided')
    assert source, ValueError('Source tables must be set')
    assert destination, ValueError('Destination tables must be set')
    assert the_proxy, ValueError('Proxy should be set')
    assert the_transaction, ValueError('Transaction is missed')

    cdef bytes the_pool = _to_binary_if_string_type(pool)
    cdef bytes the_title = _to_binary_if_string_type(title)
    cdef bytes the_script_name = _to_binary_if_string_type(script_name)
    cdef bytes the_mapper_state = _to_binary_if_string_type(mapper_state)
    cdef bytes the_reducer_state = _to_binary_if_string_type(reducer_state)
    cdef bytes the_combiner_state = _to_binary_if_string_type(combiner_state)

    cdef TOperationArgs args
    args.Source = _paths(source)
    args.Destination = _paths(destination)
    args.State.Mapper = TBuffer(the_mapper_state, len(the_mapper_state)) if the_mapper_state else TBuffer()
    args.State.Reducer = TBuffer(the_reducer_state, len(the_reducer_state)) if the_reducer_state else TBuffer()
    args.State.Combiner = TBuffer(the_combiner_state, len(the_combiner_state)) if the_combiner_state else args.State.Reducer
    args.Files.Mapper = _paths(mapper_files)
    args.Files.Reducer = _paths(reducer_files)
    args.Proxy = the_proxy
    args.Transaction = AttachTransaction(the_proxy, TString(the_token), the_transaction)

    cdef TNode optionsSpec = convert_dict_to_tnode(spec)

    if the_pool:
        optionsSpec("pool", TNode(the_pool))
    if the_title:
        optionsSpec("title", TNode(the_title))

    cdef TNode annotations
    if the_script_name:
        annotations = convert_dict_to_tnode({'script_name': the_script_name})
        optionsSpec("annotations", annotations)

    args.OptionsSpec = optionsSpec
    return args


cdef log_operation_url(proxy, operation_id):
    logger.info('Operation started %s',
        "http://{proxy}/#page=operation&mode=detail&id={id}&tab=details".format(
            proxy=proxy,
            id=operation_id,
        )
    )


cdef wait(const TOperationArgs& args, const TGUID& guid, sync=True):
    operation_id = GetGuidAsString(guid)
    if not sync:
        return operation_id
    log_operation_url(args.Proxy, operation_id)
    with nogil:
        while True:
            if IsOperationComplete(args.Transaction, guid):
                with gil:
                    return operation_id
            with gil:
                PyErr_CheckSignals()
                time.sleep(1)


def job_by_name(name):
    name = _to_binary_if_string_type(name)
    if not IsRegistered(name):
        raise ValueError('Job %s is not registered' % (name))
    return name


def run_native_map(
    mapper_name,
    source,
    destination,
    proxy,
    transaction,
    token=None,
    pool=None,
    title=None,
    state=None,
    dict spec=None,
    bool sync=True,
    script_name=None,
    **kwargs
):
    initialize(sys.argv)
    spec = spec or dict()
    cdef TString _mapper_name = mapper_name
    cdef TOperationArgs args = _prepare_args(
        source,
        destination,
        proxy=proxy,
        transaction=transaction,
        token=token,
        pool=pool,
        title=title or mapper_name,
        script_name=script_name or mapper_name,
        mapper_state=state,
        reducer_state=None,
        combiner_state=None,
        spec=spec,
        mapper_files=kwargs.get('mapper_files', []),
        reducer_files=kwargs.get('reducer_files', []),
    )
    cdef TGUID operation_id
    with nogil:
        operation_id = RunMap(_mapper_name, args)
    return wait(args, operation_id, sync)


def run_native_reduce(
    reducer_name,
    source,
    destination,
    reduce_by,
    proxy,
    transaction,
    sort_by=None,
    token=None,
    pool=None,
    title=None,
    reducer_state=None,
    dict spec=None,
    bool sync=True,
    script_name=None,
    **kwargs
):
    initialize(sys.argv)
    spec = spec or dict()
    if sort_by is None:
        sort_by = reduce_by
    cdef TString _reducer_name = reducer_name
    cdef TVector[TString] _sort_by = list_to_TVector(sort_by)
    cdef TVector[TString] _reduce_by = list_to_TVector(reduce_by)
    cdef TOperationArgs args = _prepare_args(
        source,
        destination,
        proxy=proxy,
        transaction=transaction,
        token=token,
        pool=pool,
        title=title or reducer_name,
        script_name=script_name or reducer_name,
        mapper_state=None,
        reducer_state=reducer_state,
        combiner_state=None,
        spec=spec,
        mapper_files=kwargs.get('mapper_files', []),
        reducer_files=kwargs.get('reducer_files', []),
    )
    cdef TGUID operation_id
    with nogil:
        operation_id = RunReduce(_reducer_name, _sort_by, _reduce_by, args)
    return wait(args, operation_id, sync)


def run_native_map_reduce(
    mapper_name,
    reducer_name,
    source,
    destination,
    reduce_by,
    sort_by,
    proxy,
    transaction,
    token=None,
    pool=None,
    title=None,
    mapper_state=None,
    reducer_state=None,
    dict spec=None,
    bool sync=True,
    script_name=None,
    **kwargs
):
    initialize(sys.argv)
    spec = spec or dict()
    cdef TString _mapper_name = mapper_name
    cdef TString _reducer_name = reducer_name
    cdef TVector[TString] _reduce_by = list_to_TVector(reduce_by)
    cdef TVector[TString] _sort_by = list_to_TVector(sort_by)
    cdef TOperationArgs args = _prepare_args(
        source,
        destination,
        proxy=proxy,
        transaction=transaction,
        token=token,
        pool=pool,
        title=title or (_to_binary_if_string_type(mapper_name) + _to_binary_if_string_type('_') + _to_binary_if_string_type(reducer_name)),
        script_name=script_name or (_to_binary_if_string_type(mapper_name) + _to_binary_if_string_type('_') + _to_binary_if_string_type(reducer_name)),
        mapper_state=mapper_state,
        reducer_state=reducer_state,
        combiner_state=None,
        spec=spec,
        mapper_files=kwargs.get('mapper_files', []),
        reducer_files=kwargs.get('reducer_files', []),
    )
    cdef TGUID operation_id
    with nogil:
        operation_id = RunMapReduce(_mapper_name, _reducer_name, _reduce_by, _sort_by, args)
    return wait(args, operation_id, sync)


def run_native_map_reduce_with_combiner(
    mapper_name,
    combiner_name,
    reducer_name,
    source,
    destination,
    reduce_by,
    sort_by,
    proxy,
    transaction,
    token=None,
    pool=None,
    title=None,
    mapper_state=None,
    reducer_state=None,
    combiner_state=None,
    dict spec=None,
    sync=True,
    script_name=None,
    **kwargs
):
    initialize(sys.argv)
    spec = spec or dict()
    cdef TString _mapper_name = mapper_name
    cdef TString _combiner_name = combiner_name
    cdef TString _reducer_name = reducer_name
    cdef TVector[TString] _reduce_by = list_to_TVector(reduce_by)
    cdef TVector[TString] _sort_by = list_to_TVector(sort_by)
    cdef TOperationArgs args = _prepare_args(
        source,
        destination,
        proxy=proxy,
        transaction=transaction,
        token=token,
        pool=pool,
        title=title or (mapper_name + '_' + combiner_name + '_' + reducer_name),
        script_name=script_name or (mapper_name + '_' + combiner_name + '_' + reducer_name),
        mapper_state=mapper_state,
        reducer_state=reducer_state,
        combiner_state=combiner_state,
        spec=spec,
        mapper_files=kwargs.get('mapper_files', []),
        reducer_files=kwargs.get('reducer_files', []),
    )
    cdef TGUID operation_id
    with nogil:
        operation_id = RunMapReduceWithCombiner(_mapper_name, _combiner_name, _reducer_name, _reduce_by, _sort_by, args)
    return wait(args, operation_id, sync)


def run_native_join_reduce(
    reducer_name,
    source,
    destination,
    join_by,
    proxy,
    transaction,
    token=None,
    pool=None,
    title=None,
    reducer_state=None,
    dict spec=None,
    bool sync=True,
    script_name=None,
    **kwargs
):
    initialize(sys.argv)
    spec = spec or dict()
    cdef TString _reducer_name = reducer_name
    cdef TVector[TString] _join_by = list_to_TVector(join_by)
    cdef TOperationArgs args = _prepare_args(
        source,
        destination,
        proxy=proxy,
        transaction=transaction,
        token=token,
        pool=pool,
        title=title or reducer_name,
        script_name=script_name or reducer_name,
        mapper_state=None,
        reducer_state=reducer_state,
        combiner_state=None,
        spec=spec,
        mapper_files=kwargs.get('mapper_files', []),
        reducer_files=kwargs.get('reducer_files', []),
    )
    cdef TGUID operation_id
    with nogil:
        operation_id = RunJoinReduce(_reducer_name, _join_by, args)
    return wait(args, operation_id, sync)


def initialize(args):
    args = [_to_binary_if_string_type(arg) for arg in args]
    assert len(args) < 1024

    cdef int argc = 0;
    cdef char* argv[1024]

    for arg in args:
        argv[argc] = arg
        argc += 1

    with nogil:
        Initialize(argc, <const char**>argv)
