import more_itertools

from mapreduce.yt.interface.protos.extension_pb2 import (
    column_name,
    key_column_name,
)


def get_column_name(field):
    options = field.GetOptions()
    ext_column_name = options.Extensions[column_name]
    ext_key_column_name = options.Extensions[key_column_name]
    return ext_column_name or ext_key_column_name or field.name


def get_column_type(field):
    assert field.label != field.LABEL_REPEATED
    return {
        field.CPPTYPE_INT32: 'int64',
        field.CPPTYPE_INT64: 'int64',
        field.CPPTYPE_UINT32: 'uint64',
        field.CPPTYPE_UINT64: 'uint64',
        field.CPPTYPE_FLOAT: 'double',
        field.CPPTYPE_DOUBLE: 'double',
        field.CPPTYPE_BOOL: 'boolean',
        field.CPPTYPE_STRING: 'string',
        field.CPPTYPE_MESSAGE: 'string',
        field.CPPTYPE_ENUM: 'string',
    }.get(field.cpp_type)


def is_key_column(field):
    return bool(field.GetOptions().Extensions[key_column_name])


def create_schema(proto, dynamic=False, strong=False, **kwargs):
    descriptor = proto.DESCRIPTOR
    schema = []

    for field in descriptor.fields:
        field_name = get_column_name(field)
        field_type = get_column_type(field)
        element = dict(name=field_name, type=field_type)
        if strong and is_key_column(field):
            element['sort_order'] = 'ascending'
            if not dynamic:
                element['required'] = 'true'
        element.update(kwargs.get(field_name, {}))
        schema.append(element)

    # sort columns should go first
    without_sort_order, with_sort_order = more_itertools.partition(lambda el: el.get('sort_order') is not None, schema)

    schema = list(with_sort_order) + list(without_sort_order)

    return schema


class Fields(object):
    def __init__(self, proto):
        descriptor = proto.DESCRIPTOR
        for field in descriptor.fields:
            field_name = get_column_name(field)
            self.__setattr__(field_name, field_name)


def extract_fields(proto):
    return Fields(proto)
