from functools import partial

from crypta.lib.python.purgatory.common import wrap_path, get_value


class ConversionError(Exception):
    pass


def put_value(record, path, value):
    if not isinstance(record, dict):
        raise ConversionError("Object {} is not a dict".format(record))

    iterator = record
    subpath = []
    for field in path[:-1]:
        subpath.append(field)
        if field not in iterator:
            iterator[field] = dict()
        if not isinstance(iterator[field], dict):
            raise ConversionError("Subpath {} of path {} is present in {} and doesn't point to dict".format(
                                  subpath, path, record))

        iterator = iterator[field]

    iterator[path[-1]] = value
    return record


def convert_and_store(src_dict, dst_dict, src_path, dst_path, convert=lambda x: x, optional=False):
    src_path = wrap_path(src_path)
    found, value = get_value(src_dict, src_path)
    if found:
        value = convert(value)
    elif optional:
        value = None
    else:
        raise ConversionError("Path '{}' is not present in record {}".format(src_path, src_dict))
    put_value(dst_dict, wrap_path(dst_path), value)


class Convertor(object):
    def __init__(self, convertor=None):
        if not convertor:
            self.convertors = []
        else:
            self.convertors = list(convertor.convertors)

    def convert(self, src, dst=None):
        if not dst:
            dst = dict()
        for op in self.convertors:
            op(src, dst)
        return dst

    def add(self, src_path, dst_path=None, convert=lambda x: x, optional=False):
        if not dst_path:
            dst_path = src_path
        self.convertors.append(partial(convert_and_store, src_path=src_path, dst_path=dst_path, convert=convert, optional=optional))
        return self

    def add_optional(self, src_path, dst_path=None, convert=lambda x: x):
        return self.add(src_path, dst_path, convert, True)
