def test_simple_example():
    import functools
    import six

    from crypta.lib.python.purgatory import pipeline
    from crypta.lib.python.purgatory import validation

    # 1. Define validators
    # any -> bool
    def validate_units(value, unit_regex):
        return validation.validate_regex(value, r"^[-+]?\d+\.?\d* {}$".format(unit_regex))

    validate_length = functools.partial(validate_units, unit_regex=r"[numcdk]?m")
    validate_mass = functools.partial(validate_units, unit_regex=r"[numk]?g")

    # 2. Define convertors.
    # Convertor must not check if values are valid, values must be checked in validators.
    # any -> any
    def unit_convertor(value, unit_table, dst_unit):
        number, suffix = value.split(" ")
        return "{} {}".format(float(number) * unit_table[suffix], dst_unit)

    prefixes = {"n": 1e-9, "u": 1e-6, "m": 1e-3, "c": 0.01, "d": 0.1, "": 1., "k": 1e+3}
    meters_to_inches = {"{}m".format(k): 0.0254*v for k, v in six.iteritems(prefixes)}
    grams_to_pounds = {"{}g".format(k): 453.59237*v for k, v in six.iteritems(prefixes)}

    convert_length = functools.partial(unit_convertor, unit_table=meters_to_inches, dst_unit="in")
    convert_mass = functools.partial(unit_convertor, unit_table=grams_to_pounds, dst_unit="lb")

    # 3. Make a pipeline
    metric_to_imperial = pipeline.Pipeline()
    metric_to_imperial.add("name", validate=lambda x: isinstance(x, str))
    metric_to_imperial.add_optional("length", validate=validate_length, convert=convert_length)
    metric_to_imperial.add_optional("mass", validate=validate_mass, convert=convert_mass)

    # 4. Validate and convert records
    # At first all validators are ran. If input record is valid, then the values of fields are converted.
    # If input record is invalid and raise_exception=True, then the exception is raised with the reason of failure in the message (default)
    # If input record is invalid and raise_exception=False, then None will be returned.
    inputs = [
        {"name": "George", "length": "7 cm", "mass": "173 g"},
        {"name": "John", "length": "21 cm"},  # optional fields will be filled with None
        {"name": "Anna"},
        {"length": "19 mm", "mass": "7 g"},  # invalid, because "name" is a required field
        {"name": "Pinocchio", "length": "45.3"}  # invalid, because format of "length" is invalid
    ]

    ref = [
        {"name": "George", "length": "0.001778 in", "mass": "78471.48001 lb"},
        {"name": "John", "length": "0.005334 in", "mass": None},
        {"name": "Anna", "length": None, "mass": None},
        None,
        None
    ]

    outputs = [metric_to_imperial.process(x, raise_exception=False) for x in inputs]
    assert outputs == ref
