import logging

from crypta.lib.python.purgatory.convertor import Convertor
from crypta.lib.python.purgatory.validation import Validator

logger = logging.getLogger(__name__)


class Pipeline(object):
    """
    Pipeline validates the values first, then converts them
    """
    def __init__(self, validator=None, convertor=None):
        """
        :param validator: `~crypta.lib.python.purgatory.validation.Validator`, by default an empty one.
        :param convertor: `~crypta.lib.python.purgatory.convertor.Convertor`, by default an empty one.
        """
        if validator:
            self.validator = validator
        else:
            self.validator = Validator()

        if convertor:
            self.convertor = convertor
        else:
            self.convertor = Convertor()

    def process(self, src, dst=None, raise_exception=True):
        """
        Validate and convert value

        :param src: source value
        :param dst: destination value, by default constructs an empty dict
        :param raise_exception: if True, will raise exception if `src` is invalid, else will just return `None`
            immediately after failed validation
        :return: `None` or converted value.
        """
        if raise_exception:
            self.validator.raise_if_invalid(src)
        else:
            valid, errors = self.validator(src)
            if not valid:
                for error in errors:
                    logger.error(error)
                return None

        if not dst:
            dst = dict()
        return self.convertor.convert(src, dst)

    def add(self, src_path, dst_path=None, validate=lambda x: True, convert=lambda x: x, optional=False):
        """
        A shortcut for adding validation and conversion of "single field -> single field".
        For a more complex logic add validators and convertors directly to `pipeline.validator` and `pipeline.convertor`

        :param src_path: a string or a list of strings, path to value in the source dict
        :param dst_path: a string or a list of strings, path to value in the destination dict, by default is the same as `src_path`
        :param validate: a callable with single parameter that returns boolean value ('is object valid?') and optional list of errors.
            By default accepts all values as valid.
        :param convert:  a callable with single parameter that must return a converted value. By default returns the
            same value as in source dict.
        :param optional: if True, field is considered optional
        :return: self
        """
        self.validator.add(src_path, validate, optional)
        self.convertor.add(src_path, dst_path, convert, optional)
        return self

    def add_optional(self, src_path, dst_path=None, validate=lambda x: True, convert=lambda x: x):
        """
        A shortcut for adding an optional validation and conversion of "single field -> single field".
        For a more complex logic add validators and convertors directly to `pipeline.validator` and `pipeline.convertor`

        :param src_path: a string or a list of strings, path to value in the source dict
        :param dst_path: a string or a list of strings, path to value in the destination dict, by default is the same as `src_path`
        :param validate: a callable with single parameter that returns boolean value ('is object valid?') and optional list of errors.
            By default accepts all values as valid.
        :param convert:  a callable with single parameter that must return a converted value. By default returns the
            same value as in source dict.
        :return: self
        """
        return self.add(src_path, dst_path, validate, convert, True)

    def get_errors(self, value):
        """
        Return validation errors for value
        :param value: value to validate
        :return: list of errors
        """
        return self.validator(value)[1]
