import functools
import json
import six
import re
from six.moves import collections_abc

from crypta.lib.python.purgatory.common import wrap_path, get_value

LAT_LIMIT = 90.
LON_LIMIT = 180.


class ValidationError(Exception):
    pass


def validate_regex(value, regex):
    return isinstance(value, six.string_types) and re.match(regex, value) is not None


def validate_uint(value):
    return isinstance(value, int) and value >= 0


def validate_uint_range(value, min, max):
    if not validate_uint(value):
        return False
    return min <= value <= max


def validate_unixtime(value):
    return validate_uint_range(value, 10**9, 10**10)


def validate_float(value):
    return isinstance(value, float) or isinstance(value, int)


def validate_float_range(value, min, max):
    if not validate_float(value):
        return False
    return min <= value <= max


def validate_lat(value):
    return validate_float_range(value, -LAT_LIMIT, LAT_LIMIT)


def validate_lon(value):
    return validate_float_range(value, -LON_LIMIT, LON_LIMIT)


def validate_list(value, validate=lambda x: True, allow_empty=True):
    if not isinstance(value, list):
        return False
    if not value:
        return allow_empty
    return all(validate(item) for item in value)


def validate_dict(value):
    return isinstance(value, dict)


def validate_or(value, validators=list()):
    return any(validate(value) for validate in validators)


def make_type_validator(type_):
    def wrapper(value):
        return isinstance(value, type_)

    wrapper.__name__ = "validate_{}".format(type_.__name__)
    return wrapper


class PathValidatorError(object):
    def __init__(self, path, record, validator=None, subpath=None):
        self.path = path
        self.record = record
        self.validator = validator
        self.subpath = subpath

    @staticmethod
    def __render(arg):
        if callable(arg):
            return PathValidatorError.__make_func_name(arg)
        elif isinstance(arg, dict):
            return repr({k: PathValidatorError.__render(v) for k, v in six.iteritems(arg)})
        elif isinstance(arg, collections_abc.Iterable):
            return arg.__class__(PathValidatorError.__render(x) for x in arg)
        else:
            return arg

    @staticmethod
    def __make_func_name(func):
        name = "unknown"
        if hasattr(func, "__name__"):
            name = func.__name__
        elif isinstance(func, functools.partial):
            name = "func='{name}', args={args}, kwargs={keywords}".format(
                name=PathValidatorError.__make_func_name(func.func),
                args=PathValidatorError.__render(func.args),
                keywords=PathValidatorError.__render(func.keywords)
            )
        return name

    @staticmethod
    def __stringify_path(path):
        return ".".join(path)

    @property
    def validator_name(self):
        if self.subpath:
            if self.subpath == self.path:
                return "'{}' is not present".format(self.subpath_string)
            else:
                return "'{}' is not present or is not a dict".format(self.subpath_string)
        else:
            return self.__make_func_name(self.validator)

    @property
    def subpath_string(self):
        return self.__stringify_path(self.subpath)

    @property
    def path_string(self):
        return self.__stringify_path(self.path)

    @property
    def value_in_path(self):
        valid, value_or_subpath = get_value(self.record, self.path)
        if valid:
            return value_or_subpath
        else:
            return None

    @property
    def record_json(self):
        return json.dumps(self.record)

    def __repr__(self):
        return "Invalid value. Validator: <{}>, path: {}, value: {}".format(self.validator_name, self.path_string, self.record)


class PathValidator(object):
    def __init__(self, path, validator, optional):
        self.path = wrap_path(path)
        self.validator = validator
        self.optional = optional

    def __call__(self, obj):
        found, value_or_subpath = get_value(obj, self.path)
        if not found:
            if self.optional:
                return True, []
            else:
                return False, [PathValidatorError(self.path, obj, subpath=value_or_subpath)]

        result = self.validator(value_or_subpath)
        if isinstance(result, bool):
            if result:
                return True, []
            else:
                return False, [PathValidatorError(self.path, obj, validator=self.validator)]
        else:
            return result


class Validator(object):
    def __init__(self):
        self.validators = []

    def __call__(self, obj):
        """
        Call validation handlers and collect all errors.

        :param obj: object to validate
        :return: tuple: (<object is valid?>, <list of errors>)
        """
        valid = True
        errors = []
        for validator in self.validators:
            path_valid, path_errors = validator(obj)
            errors.extend(path_errors)
            valid = valid and path_valid

        return valid, errors

    def validate(self, obj):
        """
        Call validation handlers and collect all errors.

        :param obj: object to validate
        :return: True if obj is valid, False if invalid.
        """
        return self(obj)[0]

    def raise_if_invalid(self, obj):
        """
        Call validation handlers. If any errors occured, raise an exception.

        :param obj: object to validate
        :raises ValidationError if value is invalid
        """
        valid, errors = self(obj)
        if not valid:
            raise ValidationError("Object is invalid: {}".format("; ".join(repr(err) for err in errors)))

    def add(self, path, validator=lambda x: True, optional=False):
        """
        Add a path validator

        :param path: a string or a list of strings, path to the value in a tree
        :param validator: a callable that accepts single parameter - value to validate. Must return a boolean -
            result of validation. May additionally return list of errors
        :param optional: if True, object will still be valid if given path does not exist in it
        :return: self
        """
        self.validators.append(PathValidator(path, validator, optional))
        return self

    def add_optional(self, path, validator=lambda x: True):
        """
        Add an optional path validator

        :param path: a string or a list of strings, path to the value in a tree
        :param validator: a callable that accepts single parameter - value to validate. Must return a boolean -
            result of validation. May additionally return list of errors
        :return: self
        """
        return self.add(path, validator, True)
