# -*- coding: utf-8 -*-

import requests
from retry import retry


class RetryableHttpClientError(Exception):
    def __init__(self, message, text):
        super(RetryableHttpClientError, self).__init__(message)
        self.text = text


class RetryableHttpClient(object):
    def __init__(self, tries_count, delay, jitter=0, timeout_in_secs=60, logger=None):
        self.tries_count = tries_count
        self.delay = delay
        self.jitter = jitter
        self.timeout_in_secs = timeout_in_secs
        self.logger = logger

    def _make_request_with_retries(self, method, url, params=None, data=None,
                                   headers=None, cookies=None, files=None, json=None,
                                   expected_status_codes=(requests.codes.ok, requests.codes.bad_request,)):

        def get_exception(response):
            message = 'Request failed: {method} {url}. Status code: {code} {reason}. {text}'.format(
                method=method,
                url=url,
                code=response.status_code,
                reason=response.reason,
                text=response.text.encode('utf-8'),
            )
            return RetryableHttpClientError(message, response.text)

        if method not in ('GET', 'HEAD', 'POST', 'PUT', 'DELETE', 'PATCH'):
            raise RetryableHttpClientError('Unknown method: {method}.'.format(method=method))

        kwargs = {}
        if params is not None:
            kwargs['params'] = params
        if data is not None:
            kwargs['data'] = data
        if headers is not None:
            kwargs['headers'] = headers
        if cookies is not None:
            kwargs['cookies'] = cookies
        if files is not None:
            kwargs['files'] = files
        if json is not None:
            kwargs['json'] = json

        @retry(exceptions=(RetryableHttpClientError, requests.exceptions.ConnectionError), tries=self.tries_count, delay=self.delay, jitter=self.jitter)
        def make_request():
            response = requests.request(method, url, timeout=self.timeout_in_secs, **kwargs)

            if response.status_code not in expected_status_codes:
                e = get_exception(response)
                if self.logger:
                    self.logger.warn(e)

                raise e

            return response

        result_response = make_request()
        if result_response.status_code != requests.codes.ok:
            raise get_exception(result_response)

        return result_response

    def _make_get_request(self, url, params=None, data=None, headers=None,
                          cookies=None):
        return self._make_request_with_retries(
            method='GET',
            url=url,
            params=params,
            data=data,
            headers=headers,
            cookies=cookies,
        )

    def _make_post_request(self, url, params=None, data=None, headers=None,
                           cookies=None, files=None):
        return self._make_request_with_retries(
            method='POST',
            url=url,
            params=params,
            data=data,
            headers=headers,
            cookies=cookies,
            files=files,
        )

    def _make_put_request(self, url, data=None, params=None, headers=None,
                          cookies=None):
        return self._make_request_with_retries(
            method='PUT',
            url=url,
            params=params,
            data=data,
            headers=headers,
            cookies=cookies,
        )

    def _make_delete_request(self, url, data=None, params=None, headers=None,
                             cookies=None):
        return self._make_request_with_retries(
            method='DELETE',
            url=url,
            params=params,
            data=data,
            headers=headers,
            cookies=cookies,
        )
