import json

import requests
from rtmapreduce.protos.query_pb2 import (
    TGetOperationPreviewResponse,
    TGetOperationStateResponse,
    TGetRecordsResponse,
    TGetTablePreviewResponse,
)
from rtmapreduce.protos.configuration_pb2 import TGetOperationsResponse

from crypta.lib.python.rtmr.model import data


DATA_REQUEST = "http://{host}:8080/api/v2/data.proto"
STATE_REQUEST = "http://{host}:8080/api/v2/state.proto"
PREVIEW_REQUEST = "http://{host}:8080/api/v2/preview.proto"
OPERATIONS_REQUEST = "http://{host}:8080/api/v2/operations.proto"


class Client(object):
    def __init__(self, hostname, parser_config=None, max_time_back=86400):
        self._hostname = hostname
        self._parser_config = parser_config or {}
        self._max_time_back = max_time_back

    def _get_parser(self, parser_name):
        return self._parser_config.get(parser_name, lambda x: x)

    def _parse(self, parser_name, value):
        return self._get_parser(parser_name)(value)

    def _parse_chunk(self, chunk):
        key = chunk.Key
        parser = self._get_parser(chunk.Table)
        return [data.Record(key, record.SubKey, parser(record.Value)) for record in chunk.Entries]

    def _parse_entries(self, entries):
        return [
            {
                "key": entry.Key,
                "subkey": entry.SubKey,
                "value": self._parse(entry.Table, entry.Value),
                "table": entry.Table
            }
            for entry in entries
        ]

    def _make_data_subrequests(self, table, keys):
        return [json.dumps({
            "Table": table,
            "Key": key,
            "MaxTimeBack": self._max_time_back
        }) for key in keys]

    def read_table(self, table_path, keys):
        request = DATA_REQUEST.format(host=self._hostname)
        response = requests.get(request, stream=True, params={"request": self._make_data_subrequests(table_path, keys)})
        response.raise_for_status()

        proto_resp = TGetRecordsResponse()
        proto_resp.ParseFromString(response.content)

        result = []

        for chunk in proto_resp.Chunks:
            result += self._parse_chunk(chunk)

        return result

    def read_state(self, operation, key):
        request = STATE_REQUEST.format(host=self._hostname)
        response = requests.get(request, stream=True, params={"path": operation, "id": key})
        response.raise_for_status()

        proto_resp = TGetOperationStateResponse()
        proto_resp.ParseFromString(response.content)

        return self._parse(operation, proto_resp.State)

    def preview_table(self, table):
        request = PREVIEW_REQUEST.format(host=self._hostname)
        response = requests.get(request, stream=True, params={"table": table})
        response.raise_for_status()

        proto_resp = TGetTablePreviewResponse()
        proto_resp.ParseFromString(response.content)

        result = []

        for record in proto_resp.Records:
            result.append({
                "timestamp": record.Timestamp,
                "update": self._parse_chunk(record.Update)
            })

        return result

    def preview_operation(self, operation, count=10):
        request = PREVIEW_REQUEST.format(host=self._hostname)
        response = requests.get(request, stream=True, params={"operation": operation, "count": count})
        response.raise_for_status()

        proto_resp = TGetOperationPreviewResponse()
        proto_resp.ParseFromString(response.content)

        result = []

        state_parser = self._get_parser(operation)

        def parse(records):
            return state_parser("".join(records))

        for record in proto_resp.Records:
            result.append({
                "timestamp": record.Timestamp,
                "input": self._parse_entries(record.Input),
                "output": self._parse_entries(record.Output),
                "prev_state": parse(record.PrevState),
                "new_state": parse(record.NewState)
            })

        return result

    def get_operation_config(self, operation_name):
        request = OPERATIONS_REQUEST.format(host=self._hostname)
        response = requests.get(request, stream=True, params={"filter": "^{}$".format(operation_name)})
        response.raise_for_status()

        proto_resp = TGetOperationsResponse()
        proto_resp.ParseFromString(response.content)

        for operation in proto_resp.Operations:
            if operation.Name == operation_name:
                return operation

        return None
