import json

from crypta.lib.python.rtmr.model import data
import six


def make_encoder(parse_value=None):
    parse_value = parse_value or (lambda x: six.ensure_text(x, encoding="utf-8"))

    class Encoder(json.JSONEncoder):
        def default(self, obj):
            if isinstance(obj, data.Record):
                return {
                    "key": six.ensure_text(obj.key, encoding="utf-8"),
                    "subkey": six.ensure_text(obj.subkey, encoding="utf-8"),
                    "value": parse_value(obj.value)
                }
            elif isinstance(obj, data.TableSwitch):
                return {
                    "table": obj.table
                }
            elif isinstance(obj, six.binary_type):
                return six.ensure_text(obj)
            else:
                raise Exception("Unknown object type {}".format(type(obj)))
    return Encoder


def make_decoder(serialize_value=None):
    serialize_value = serialize_value or (lambda x: x)

    class Decoder(json.JSONDecoder):
        record_keys = frozenset(("key", "subkey", "value"))
        table_switch_keys = frozenset(("table",))

        def __init__(self, *args, **kwargs):
            json.JSONDecoder.__init__(self, object_hook=self.object_hook, *args, **kwargs)

        def object_hook(self, dct):
            keys_set = set(dct.keys())
            if self.record_keys <= keys_set:
                return data.Record(dct["key"], dct["subkey"], serialize_value(dct["value"]))
            elif keys_set == self.table_switch_keys:
                return data.TableSwitch(dct["table"])

            return dct

    return Decoder
