import os
import struct

from crypta.lib.python.rtmr.model.data import (
    Record,
    TableSwitch
)
import six


class Reader(object):
    def __init__(self, table):
        self.table = table
        self.stream = None

    def __enter__(self):
        self.stream = open(self.table, 'rb')
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stream.close()
        return

    def _read_len(self):
        raw_length = self.stream.read(4)
        length = struct.unpack("@i", raw_length)[0]
        return length

    def _read_value(self):
        length = self._read_len()

        is_table_switch = length == -1
        if is_table_switch:
            length = self._read_len()

        return is_table_switch, self.stream.read(length)

    def _read(self):
        is_table_switch, key_or_table_switch = self._read_value()
        if is_table_switch:
            return TableSwitch(key_or_table_switch)

        subkey_is_table_switch, subkey = self._read_value()
        value_is_table_switch, value = self._read_value()
        assert not subkey_is_table_switch and not value_is_table_switch

        return Record(key_or_table_switch, subkey, value)

    def __iter__(self):
        while self.stream.tell() < os.fstat(self.stream.fileno()).st_size:
            yield self._read()


class Writer(object):
    def __init__(self, table):
        self.table = table
        self.stream = None

    def __enter__(self):
        self.stream = open(self.table, 'wb')
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stream.close()
        return

    def _write_len(self, length):
        raw_length = struct.pack("@i", length)
        return self.stream.write(raw_length)

    def _write_value(self, value):
        if isinstance(value, six.text_type):
            value = value.encode("utf-8")

        self._write_len(len(value))
        return self.stream.write(value)

    def write(self, record_or_table_switch):
        if isinstance(record_or_table_switch, Record):
            self._write_value(record_or_table_switch.key)
            self._write_value(record_or_table_switch.subkey)
            self._write_value(record_or_table_switch.value)
        elif isinstance(record_or_table_switch, TableSwitch):
            self._write_len(-1)
            self._write_value(record_or_table_switch.table)
