import copy
import json
import logging
import os
import shutil
import tarfile
import tempfile
import six.moves.urllib.parse as urlparse

import requests

try:
    from sandbox import common
    from sandbox.common import rest
    from sandbox.common.proxy import OAuth
    import sandbox.common.types.misc as ctm
except ImportError:  # system python support (when sandbox package is installed with pip)
    from sandbox_common import rest
    from sandbox_common.proxy import OAuth
    import sandbox_common as common
    import sandbox_common.types.misc as ctm


PROXY_BASE_URL = "http://proxy.sandbox.yandex-team.ru"

logger = logging.getLogger(__name__)


def _extract(archive_path, filename, target_path):
    tmpdir = tempfile.mkdtemp()
    with tarfile.open(archive_path, "r:gz") as tar:
        tar.extractall(path=tmpdir)

    resource_file = os.path.join(tmpdir, filename)
    logger.debug("extracted to %s, files:\n%s", tmpdir, "\n".join(os.listdir(tmpdir)))
    logger.debug("moving %s to %s", resource_file, target_path)

    os.rename(resource_file, target_path)

    shutil.rmtree(tmpdir)


def _load_file(url, stream):
    response = requests.get(url, stream=True)
    response.raise_for_status()

    for chunk in response:
        stream.write(chunk)


class SandboxClient(object):
    def __init__(self, token=None):
        self.sandbox = rest.Client() if token is None else rest.Client(auth=OAuth(token))
        self.token = token

    def _get_last_released_resource_id_for_status(self, resource_type, release_status):
        attrs = dict(released=release_status)
        response = self.sandbox.resource.read(state="READY", order=["-id"], type=resource_type, limit=1, attrs=attrs)

        logger.debug("Response: %s", json.dumps(response, indent=4))

        if not response["items"]:
            return None

        return response["items"][0]["id"]

    def get_last_released_resource_id(self, resource_type, release_type):
        STABLE = "stable"
        TESTING = "testing"
        PRESTABLE = "prestable"
        UNSTABLE = "unstable"
        RELEASE_STATUS = (STABLE, PRESTABLE, TESTING, UNSTABLE)

        assert release_type in RELEASE_STATUS

        last_released_resource_id = None
        for status in RELEASE_STATUS:
            resource_id = self._get_last_released_resource_id_for_status(resource_type, status)
            if resource_id is not None:
                if last_released_resource_id is not None:
                    last_released_resource_id = max(last_released_resource_id, resource_id)
                else:
                    last_released_resource_id = resource_id

            if status == release_type:
                break

        return last_released_resource_id

    def load_resource(self, resource_id, target_path, resource_path=None, bundle_file=None):
        url = urlparse.urljoin(PROXY_BASE_URL, "{}/{}".format(resource_id, resource_path) if resource_path else str(resource_id))

        tmp = tempfile.NamedTemporaryFile(delete=False)
        _load_file(url, tmp)
        tmp.close()

        if bundle_file is not None:
            _extract(tmp.name, bundle_file, target_path)
            os.remove(tmp.name)
        else:
            os.rename(tmp.name, target_path)

    def get_release_type(self, version):
        attrs = {attr["name"]: attr["value"] for attr in self.sandbox.resource[version].attribute.read()}
        return attrs["released"]

    def release(self, version, release_type, new_ttl=None):
        self.sandbox.resource[version].attribute["released"].update({"value": release_type})
        if new_ttl:
            self.sandbox.resource[version].attribute["ttl"].update({"value": new_ttl})

    def upload_to_sandbox(self, path, resource_type, description, owner, attributes, logger=logger, is_folder=False, alias_path=None):
        base_url = "https://sandbox.yandex-team.ru"

        def file_meta(file_name, file_path=None):
            sz = os.path.getsize(file_name)
            if file_path is None:
                file_path = os.path.basename(file_name)
            return common.upload.HTTPHandle.FileMeta(handle=open(file_name, "rb"), size=sz, name=file_path)

        def task_url(task_id):
            return urlparse.urljoin(base_url, "task/{}".format(task_id))

        def resource_url(res_id):
            return urlparse.urljoin(base_url, "resource/{}".format(res_id))

        resource_meta = common.upload.HTTPHandle.ResourceMeta(
            type=resource_type,
            arch=ctm.OSFamily.ANY,
            owner=owner,
            description=description,
            attributes=attributes,
            release_to_yd=False,
        )

        files = []
        if is_folder:
            file_names = os.listdir(path)
            for file_name in file_names:
                files.append(file_meta(os.path.join(path, file_name), os.path.join(alias_path or "root", file_name)))
        else:
            files.append(file_meta(path, alias_path))

        uploader = common.upload.HTTPHandle(
            resource_meta, self.token, base_url, PROXY_BASE_URL, 30, *files
        )
        state, last_state, last_state_copy, last_task_state = (None,) * 4

        resource_id = None
        for state in uploader():
            if last_state != state:
                if isinstance(last_state, ctm.Upload.Check):
                    logger.info(
                        "%s file%s (%s) to upload",
                        last_state.amount, "s" if last_state.amount > 1 else "", common.utils.size2str(last_state.size)
                    )
                if isinstance(state, ctm.Upload.Check):
                    logger.info("Calculating total files size")
                if isinstance(state, ctm.Upload.Prepare):
                    logger.info("Preparing upload task")
                if isinstance(state, ctm.Upload.Share):
                    logger.info("Sharing uploaded data")
            else:
                if isinstance(state, ctm.Upload.Prepare):
                    if last_state_copy.task_id != state.task_id:
                        logger.info(
                            "Uploading task #%s created: %s", state.task_id, task_url(state.task_id)
                        )
                    if last_state_copy.resource_id != state.resource_id:
                        resource_id = state.resource_id
                        url = resource_url(state.resource_id)
                        logger.info("Resource #%s registered: %s", state.resource_id, url)
                if isinstance(state, ctm.Upload.Share):
                    if last_task_state != state.task_state:
                        logger.info("Task is in %s state", state.task_state)
                        last_task_state = state.task_state

            last_state = state
            last_state_copy = copy.deepcopy(state)

        if isinstance(state, ctm.Upload.Share):
            if last_task_state != state.task_state:
                logger.info("Task state: '%s'", state.task_state)
            if state.md5sum:
                logger.info("MD5 checksum = %s", state.md5sum)
        return resource_id

    def get_rbtorrent(self, resource_id):
        return self.sandbox.resource[resource_id].read()["skynet_id"]


def get_sandbox_client(sandbox_token=None):
    sandbox_token = os.getenv("SANDBOX_TOKEN", sandbox_token)
    assert sandbox_token is not None,\
        "No sandbox_token is specified, and there is no environment variable SANDBOX_TOKEN"

    return SandboxClient(sandbox_token)
