#!/usr/bin/env python
# -*- coding: utf-8 -*-
import collections
import logging
import os

import retry
import six

from library.python.vault_client import instances, errors


logger = logging.getLogger(__name__)

Secret = collections.namedtuple('Secret', ['secret_id', 'secret_key'])


class SecretManager(object):
    def __init__(self, secret_name_to_secret_info, secrets_by_name=None, oauth_token=None):
        oauth_token = oauth_token or os.getenv('YAV_TOKEN')
        assert oauth_token is not None or secrets_by_name is not None, \
            'You should specify either yav oauth token or secrets themselves'

        self.secret_name_to_secret_info = secret_name_to_secret_info

        self.yav_client = None
        self.secrets_by_name = {}
        if secrets_by_name is not None:
            logger.info('Using secrets from dict')
            self.secrets_by_name = secrets_by_name
        else:
            self.yav_client = instances.Production(authorization=oauth_token.strip())
            logger.info('Using secret from vault')
            self._load_all_secrets()

    @retry.retry(errors.ClientError, tries=10, delay=1, backoff=2)
    def _load_secret(self, secret_info):
        return self.yav_client.get_version(secret_info.secret_id)['value'][secret_info.secret_key].strip()

    def _load_all_secrets(self):
        for secret_name, secret_info in six.iteritems(self.secret_name_to_secret_info):
            logger.info('Downloading secret {}'.format(secret_name))
            self.secrets_by_name[secret_name] = self._load_secret(secret_info)
        logger.info('{} secrets were downloaded from Vault'.format(len(self.secret_name_to_secret_info)))

    def get_secret(self, key):
        if key not in self.secrets_by_name:
            raise KeyError('No such secret {}'.format(key))

        return self.secrets_by_name[key]
