import logging

import paramiko


logger = logging.getLogger(__name__)


class SftpClient(object):
    def __init__(self, host, port, key_file=None, username=None, password=None, timeout=10):
        self.host = host
        self.port = port
        self.username = username
        self.password = password
        self.timeout = timeout
        self.key_file = key_file
        self.client = None
        self.sftp = None

    @classmethod
    def from_proto(cls, proto):
        return cls(
            host=proto.Host,
            port=proto.Port,
            key_file=proto.KeyFile or None,
            username=proto.Username or None,
            password=proto.Password or None,
            timeout=proto.TimeoutSec,
        )

    def __enter__(self):
        self.client = paramiko.SSHClient()
        self.client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self.client.connect(
            hostname=self.host,
            port=self.port,
            key_filename=self.key_file,
            username=self.username,
            password=self.password,
            timeout=self.timeout,
            look_for_keys=False,
            allow_agent=False,
        )
        self.sftp = self.client.open_sftp()
        return self

    def __exit__(self, *args):
        self.sftp.close()
        self.client.close()

    def listdir(self, path='.'):
        return self.sftp.listdir(path)

    def upload(self, local_file, remote_file):
        logger.info("Upload local file %s as %s to FTP", local_file, remote_file)
        self.sftp.put(local_file, remote_file)

    def download(self, remote_file, local_file):
        logger.info("Download remote file %s as %s from FTP", remote_file, local_file)
        self.sftp.get(remote_file, local_file, prefetch=False)

    def remove(self, filename):
        logger.info("Remove %s from FTP", filename)
        self.sftp.remove(filename)

    def rename(self, oldpath, newpath):
        logger.info("Rename %s to %s", oldpath, newpath)
        self.sftp.rename(oldpath, newpath)

    def mkdir(self, path):
        logger.info("Mkdir %s", path)
        self.sftp.mkdir(path)

    def rmdir(self, path):
        logger.info("Rmdir %s", path)
        self.sftp.rmdir(path)

    def listdir_attr(self, path='.'):
        return self.sftp.listdir_attr(path)

    def list_l(self, path='.'):
        """Return str with 'ls -l' fmt"""
        for name, attrs in zip(self.listdir(path), self.listdir_attr(path)):
            yield "{:20}{}".format(name, attrs)
