import ctypes
import os
import re

import itertools
import threading
import time

from crypta.lib.python.shared_data_structures.immutable_shareable_dict import ImmutableShareableDict


class HotSwapImmutableShareableDictView:
    def __init__(self, current_name, links, id_):
        self.current_name = current_name
        self.links = links
        self.data = None
        self.id = id_

    def update_dicts(self):
        if self.data is None or self.data.name != self.current_name.value:
            if self.data is not None:
                self.data.close()
            self.data = ImmutableShareableDict(name=self.current_name.value)
            self.links[self.id] = self.current_name.value

    def get(self, key):
        self.update_dicts()
        return self.data.get(key)

    def close(self):
        if self.data:
            self.data.close()


class HotSwapImmutableShareableDict:
    def __init__(self, prefix_name, manager):
        self.prefix_name = prefix_name
        self.name_id_gen = itertools.count()
        self.view_id_gen = itertools.count()
        self.data = None
        self.to_remove = {}
        self.current_name = manager.Value(ctypes.c_wchar_p, "")
        self.remove_pages()
        self.links = manager.dict()
        self.start_periodic_cleanup()

    def update(self, data):
        if self.data is not None:
            self.to_remove[self.data.name] = self.data

        name = "{}_{}".format(self.prefix_name, next(self.name_id_gen))

        self.data = ImmutableShareableDict(data, name)
        self.current_name.value = name

    def create_view(self):
        return HotSwapImmutableShareableDictView(self.current_name, self.links, next(self.view_id_gen))

    def cleanup(self):
        present_versions = set(self.links.values())
        to_remove_ids = []
        for id_ in self.to_remove:
            if id_ not in present_versions:
                self.to_remove[id_].close()
                to_remove_ids.append(id_)

        for id_ in to_remove_ids:
            del self.to_remove[id_]

    def periodic_cleanup(self):
        while True:
            time.sleep(60)
            self.cleanup()

    def start_periodic_cleanup(self):
        self.thread = threading.Thread(target=self.periodic_cleanup)
        self.thread.daemon = True
        self.thread.start()

    def close(self):
        for data in self.to_remove.values():
            data.close()
        self.to_remove.clear()

        if self.data:
            self.data.close()

    def remove_pages(self):
        shm_path = "/dev/shm"
        for filename in os.listdir(shm_path):
            if re.fullmatch(r"{}_\d+".format(self.prefix_name), filename):
                os.remove(os.path.join(shm_path, filename))
