from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
import smtplib


class TextMailSender(object):
    def __init__(self, host, port=25, default_from_addr=None, default_to_addrs=None, default_cc=None,
                 default_bcc=None, dry_run=False):
        self._host = host
        self._port = port
        self._default_from_addr = default_from_addr
        self._default_to_addrs = default_to_addrs
        self._default_cc = default_cc or []
        self._default_bcc = default_bcc or []
        self._dry_run = dry_run

    @staticmethod
    def _generate_mail(from_addr, to_addrs, subject, text, cc, attachments):
        msg = MIMEMultipart() if attachments else MIMEText(text)
        msg['Subject'] = subject
        msg['From'] = from_addr
        msg['To'] = ", ".join(to_addrs)

        if cc:
            msg['CC'] = ", ".join(cc)

        if attachments:
            for attachment in [MIMEText(text)] + attachments:
                msg.attach(attachment)

        return msg.as_string()

    def send(self, subject, text, from_addr=None, to_addrs=None, cc=None, bcc=None, attachments=None):
        if self._dry_run:
            return

        actual_from_addr = from_addr or self._default_from_addr
        actual_to_addrs = to_addrs or self._default_to_addrs
        actual_cc = cc or self._default_cc
        actual_bcc = bcc or self._default_bcc

        if not actual_from_addr:
            raise Exception("no from_addr specified")
        if not actual_to_addrs or not all(actual_to_addrs):
            raise Exception("no to_addrs specified or some of to_addrs are invalid")

        msg = self._generate_mail(actual_from_addr, actual_to_addrs, subject, text, actual_cc, attachments)

        smtp = smtplib.SMTP(self._host, self._port)
        smtp.sendmail(actual_from_addr, actual_to_addrs + actual_cc + actual_bcc, msg)
        smtp.quit()

    @classmethod
    def from_proto(cls, proto):
        return cls(
            host=proto.Host,
            port=proto.Port,
            default_from_addr=proto.DefaultFromAddr,
            default_to_addrs=proto.DefaultToAddrs,
            default_cc=proto.DefaultCc,
            default_bcc=proto.DefaultBcc,
            dry_run=proto.DryRun,
        )
