from google.protobuf import json_format
from library.python.monlib import encoder
import requests
from six.moves import urllib

from crypta.lib.python.solomon.proto import alert_pb2


class Format:
    json = "json"
    spack = "spack"


class SolomonClient(object):
    def __init__(self, auth_token, address="https://solomon.yandex-team.ru", timeout=10):
        self.address = address
        self.session = requests.session()
        self.session.headers["Authorization"] = "OAuth {}".format(auth_token)
        self.timeout = timeout

    def _get_url(self, path):
        return urllib.parse.urljoin(self.address, path)

    def get_alert_url(self, project_id, alert_id):
        path = "/api/v2/projects/{project_id}/alerts/{alert_id}".format(project_id=project_id, alert_id=alert_id)
        return self._get_url(path)

    def get_project_alerts_url(self, project_id):
        path = "/api/v2/projects/{project_id}/alerts".format(project_id=project_id)
        return self._get_url(path)

    def list_alerts(self, project_id):
        page_token = ""

        while page_token is not None:
            response = self.session.get(self.get_project_alerts_url(project_id), params={"pageToken": page_token}, timeout=self.timeout)
            response.raise_for_status()

            response_json = response.json()
            for alert in response_json["items"]:
                yield alert

            page_token = response_json.get("nextPageToken")

    def get_alert(self, project_id, alert_id):
        response = self.session.get(self.get_alert_url(project_id, alert_id), timeout=self.timeout)
        if response.status_code == requests.codes.ok:
            alert = alert_pb2.TAlert()
            json_format.ParseDict(response.json(), alert, ignore_unknown_fields=True)
            return alert
        elif response.status_code == requests.codes.not_found:
            return None

        response.raise_for_status()

    def create_alert(self, alert):
        project_id = alert.projectId
        response = self.session.post(self.get_project_alerts_url(project_id), json=json_format.MessageToDict(alert), timeout=self.timeout)
        response.raise_for_status()

    def update_alert(self, alert):
        project_id = alert.projectId
        alert_id = alert.id

        uploaded_alert = self.get_alert(project_id, alert_id)
        if uploaded_alert is None:
            raise Exception("Alert with id '{}' not found")

        alert.version = uploaded_alert.version

        response = self.session.put(self.get_alert_url(project_id, alert_id), json=json_format.MessageToDict(alert), timeout=self.timeout)
        response.raise_for_status()

    def remove_alert(self, project_id, alert_id):
        response = self.session.delete(self.get_alert_url(project_id, alert_id), timeout=self.timeout)
        response.raise_for_status()

    def push(self, project, cluster, service, data, fmt):
        url = self._get_url("/api/v2/push")
        params = {
            "project": project,
            "cluster": cluster,
            "service": service,
        }
        headers = {
            "Content-Type": get_content_type(fmt),
        }
        headers.update(self.session.headers)
        response = self.session.post(url, data=data, params=params, headers=headers)
        response.raise_for_status()

    def push_registry(self, project, cluster, service, registry, fmt=Format.json, **kwargs):
        data = encoder.dumps(registry, format=fmt, **kwargs)
        self.push(project, cluster, service, data, fmt)


def get_content_type(fmt):
    if fmt == Format.json:
        return "application/json"
    elif fmt == Format.spack:
        return "application/x-solomon-spack"
    else:
        raise Exception("Unknown solomon data format '{}'".format(fmt))
