import os

import six.moves.urllib.parse as urlparse

from crypta.lib.python.spine.arcadia_ci import consts
from crypta.lib.python.spine.config_registry import ConfigRegistry


def get_id_from_path(path):
    return os.path.basename(path).split('.')[0]


class Release(object):
    TAG = "arcadia-ci-release"

    def __init__(self, title, abs_paths, stages, abc_service, project_title=None):
        self.title = title
        self.id = title.lower().replace(" ", "-")
        self.project_title = project_title or title
        self.filters = Filters(abs_paths)
        self.stages = stages
        self.abc_service = abc_service

    @property
    def jobs(self):
        return []

    @property
    def flow(self):
        return Flow("flow-{}".format(self.id), self.jobs)

    def render(self):
        return {
            "title": self.title,
            "auto": True,
            "stages": self.stages,
            "flow": self.flow.id,
            "filters": self.filters.render(),
        }


class Flow(object):
    def __init__(self, id, jobs):
        self.id = id
        self.jobs = jobs

    def render(self):
        return {
            "jobs": {job.id: job.render() for job in self.jobs}
        }


class Filters(object):
    def __init__(self, abs_paths=None):
        self.abs_paths = abs_paths

    def render(self):
        return [{
            "discovery": "dir",
            "abs-paths": self.abs_paths
        }]


class Job(object):
    def __init__(self, id_, title, task, stage, needs=None, input=None, manual=False):
        self.id = id_
        self.title = title
        self.task = task
        self.needs = needs
        self.input = input
        self.manual = manual
        self.stage = stage

    def render(self):
        result = {
            "title": self.title,
            "task": self.task,
            "stage": self.stage,
        }
        if self.needs:
            result["needs"] = self.needs
        if self.manual:
            result["manual"] = True
        if self.input:
            result["input"] = self.input
        return result


class BuildJob(Job):
    def __init__(self, resource_type, package_path, attrs=None, id_=None, needs=None):
        id_ = id_ or resource_type
        super(BuildJob, self).__init__(
            id_="build-{}".format(id_),
            title="Build {}".format(id_),
            task="projects/crypta/ya_package",
            stage="build",
            input={
                "packages": package_path,
                "resource_type": resource_type,
                "package_resource_attrs": attrs or {}
            },
            needs=needs,
        )
        self.resource_type = resource_type


class BuildNannyJob(BuildJob):
    pass


class BuildUniversalBundleJob(BuildJob):
    def __init__(self, package_path, needs, bundle_id=None):
        self.bundle_id = bundle_id or get_id_from_path(package_path)
        super(BuildUniversalBundleJob, self).__init__(
            id_=self.bundle_id,
            resource_type=consts.CRYPTA_UNIVERSAL_BUNDLE,
            package_path=package_path,
            attrs={
                "name": self.bundle_id,
            },
            needs=needs,
        )


class BuildDockerJob(Job):
    def __init__(self, package_path, needs):
        id_ = get_id_from_path(package_path)
        super(BuildDockerJob, self).__init__(
            id_="build-{}".format(id_),
            title="Build {}".format(id_),
            task="projects/crypta/ya_package_docker",
            stage="build",
            input={
                "packages": package_path,
                "custom_version": "${context.version_info.full}"
            },
            needs=needs,
        )
        self.resource_type = consts.CRYPTA_DOCKER


class ReleaseJob(Job):
    def __init__(self, stage, needs, build_jobs):
        super(ReleaseJob, self).__init__(
            id_="release-{}".format(stage),
            title="Release to {}".format(stage),
            task="common/releases/simple_releaser",
            stage=stage,
            needs=needs,
            input={
                "config": {
                    "deploy_system": "SANDBOX",
                    "common_release_data": {
                        "sandbox_common_release_data": {
                            "release_stage": stage,
                        },
                    },
                },
            }
        )
        self.build_jobs = build_jobs

    def render(self):
        result = super(ReleaseJob, self).render()
        result["input"]["release_items"] = [
            {
                "sandbox_release_item": {
                    "id": "${{(tasks.{}.resources[?type == '{}'])[0].id}}".format(build_job.id, build_job.resource_type),
                    "type": build_job.resource_type,
                },
            }
            for build_job in self.build_jobs
        ]
        return result


class ReleaseNannyJob(Job):
    def __init__(self, stage, needs, build_jobs):
        super(ReleaseNannyJob, self).__init__(
            id_="release-nanny-{}".format(stage),
            title="Release to {} (Nanny)".format(stage),
            task="common/releases/simple_releaser",
            stage=stage,
            needs=needs,
            input={
                "config": {
                    "deploy_system": "NANNY",
                    "wait_for_deploy": True,
                    "common_release_data": {
                        "nanny_common_release_data": {
                            "release_stage": stage,
                        },
                    },
                },
            }
        )
        self.build_jobs = build_jobs

    def render(self):
        result = super(ReleaseNannyJob, self).render()
        result["input"]["release_items"] = [
            {
                "sandbox_release_item": {
                    "id": "${{(tasks.{}.resources[?type == '{}'])[0].id}}".format(build_job.id, build_job.resource_type),
                    "type": build_job.resource_type,
                },
            }
            for build_job in self.build_jobs
        ]
        return result


class DummyJob(Job):
    def __init__(self, stage, needs, manual=False):
        super(DummyJob, self).__init__(
            id_="start-{}".format(stage),
            title="Start {}".format(stage),
            task="dummy",
            stage=stage,
            manual=manual,
            needs=needs,
        )


class RunCommandJob(Job):
    def __init__(self, title, cmd, stage, needs):
        super(RunCommandJob, self).__init__(
            id_=title.lower().replace(" ", "-"),
            title=title,
            task="common/misc/run_command",
            stage=stage,
            needs=needs,
            input={
                "config": {
                    "cmd_line": cmd,
                    "secret_environment_variables": [],
                    "arc_mount_config": {
                        "enabled": True,
                    }
                },
            }
        )

    def add_secret(self, secret_env):
        self.input["config"]["secret_environment_variables"].append({
            "key": secret_env.env,
            "secret_spec": {
                "uuid": secret_env.sec,
                "key": secret_env.id,
            },
        })


class StandardRelease(Release):
    def __init__(self, *args, **kwargs):
        self.auto_stable = kwargs.pop("auto_stable", False)
        kwargs["stages"] = [
            {
                "id": "build",
                "displace": True,
            },
            {
                "id": "testing",
                "displace": True,
            },
            {
                "id": "stable",
            }
        ]
        super(StandardRelease, self).__init__(*args, **kwargs)
        self.nanny_build_jobs = []
        self.build_jobs = []
        self.build_barrier = DummyJob(consts.BUILD, [])

    @property
    def jobs(self):
        all_build_jobs = self.nanny_build_jobs + self.build_jobs
        all_build_job_ids = [job.id for job in all_build_jobs]
        testing_barrier = DummyJob(consts.TESTING, all_build_job_ids)

        testing_jobs = []
        if self.build_jobs:
            testing_jobs.append(ReleaseJob(consts.TESTING, [testing_barrier.id], self.build_jobs))
        if self.nanny_build_jobs:
            testing_jobs.append(ReleaseNannyJob(consts.TESTING, [testing_barrier.id], self.nanny_build_jobs))

        stable_barrier = DummyJob(consts.STABLE, [job.id for job in testing_jobs], manual=not self.auto_stable)

        stable_jobs = []
        if self.build_jobs:
            stable_jobs.append(ReleaseJob(consts.STABLE, [stable_barrier.id], self.build_jobs))
        if self.nanny_build_jobs:
            stable_jobs.append(ReleaseNannyJob(consts.STABLE, [stable_barrier.id], self.nanny_build_jobs))

        return [self.build_barrier, testing_barrier, stable_barrier] + all_build_jobs + testing_jobs + stable_jobs

    def _add_build_job(self, job):
        if isinstance(job, BuildNannyJob):
            self.nanny_build_jobs.append(job)
        else:
            self.build_jobs.append(job)
        return job

    def add_docker_image(self, package_path):
        self._add_build_job(BuildDockerJob(package_path, [self.build_barrier.id]))

    def add_universal_bundle(self, package_path, bundle_id=None):
        return self._add_build_job(BuildUniversalBundleJob(package_path, [self.build_barrier.id], bundle_id=bundle_id))

    def add_bundle(self, resource_type, package_path):
        return self._add_build_job(BuildJob(resource_type, package_path, needs=[self.build_barrier.id]))

    def add_nanny_bundle(self, resource_type, package_path):
        return self._add_build_job(BuildNannyJob(resource_type, package_path, needs=[self.build_barrier.id]))


class RunCmdRelease(Release):
    def __init__(self, title, cmd, abs_paths, abc_service, project_title=None):
        super(RunCmdRelease, self).__init__(
            title=title,
            abs_paths=abs_paths,
            stages=[
                {
                    "id": "run",
                    "displace": True,
                },
            ],
            abc_service=abc_service,
            project_title=project_title,
        )
        self.run_barrier = DummyJob(consts.RUN, [])
        self.run_job = RunCommandJob(title, cmd, consts.RUN, [self.run_barrier.id])

    @property
    def jobs(self):
        return [self.run_barrier, self.run_job]


class ReleaseGenerator(ConfigRegistry):
    def __init__(self, abc_service, path_template, juggler, project_title=None):
        super(ReleaseGenerator, self).__init__()
        self.abc_service = abc_service
        self.path_template = path_template
        self.juggler = juggler
        juggler.add_subregistry(self)
        self.project_title = project_title

    def standard_release(self, *args, **kwargs):
        return self._release(StandardRelease, *args, **kwargs)

    def run_cmd_release(self, *args, **kwargs):
        return self._release(RunCmdRelease, *args, **kwargs)

    def _release(self, cls, *args, **kwargs):
        kwargs["project_title"] = self.project_title
        kwargs["abc_service"] = self.abc_service
        release = cls(*args, **kwargs)

        self.juggler.some(
            "release-{}".format(release.id)
        ).set_child(
            "ci-release-status-{}".format(self.abc_service),
            release.id,
            instance=self.path(release.project_title),
        ).add_url(
            title="Arcadia CI",
            url="https://a.yandex-team.ru/projects/{}/ci/releases/timeline?dir={}&id={}".format(
                self.abc_service,
                urlparse.quote(self.path(release.project_title)),
                release.id
            ),
            type_="doc",
        ).set_warn_limit(
            100
        ).reset_unreachable()

        return self.store(Release.TAG, release)

    def path(self, project_title):
        return self.path_template.format(project_title.replace(" ", "_"))


class CryptaReleaseGenerator(ReleaseGenerator):
    def __init__(self, juggler, project_title=None):
        super(CryptaReleaseGenerator, self).__init__("cryptadev", "crypta/spine/pushers/arcadia_ci/test/canondata/main.test_arcadia_ci_{}_", juggler, project_title)


class SpineReleaseGenerator(CryptaReleaseGenerator):
    def __init__(self, juggler):
        super(SpineReleaseGenerator, self).__init__(juggler, "Spine")
