import datetime

import six

from yp import data_model

from crypta.lib.python.spine.config_registry import ConfigRegistry
from crypta.lib.python.spine.deploy import consts
import crypta.lib.python.spine.juggler.consts as juggler_consts


class DeployStageGenerator(ConfigRegistry):
    def __init__(self, abc_service, project):
        super(DeployStageGenerator, self).__init__()
        self.abc_service = abc_service
        self.project = project

    def stage(self, *args, **kwargs):
        return self.store(Stage.TAG, Stage(self, self.abc_service, self.project, *args, **kwargs))


class Stage(object):
    TAG = "deploy.stage"

    def __init__(self, registry, abc_service, project, id_):
        self.registry = registry

        abc_service = "abc:service:{}".format(abc_service)
        self.proto = data_model.TStage(
            meta=data_model.TStageMeta(
                id=id_,
                account_id=abc_service,
                project_id=project,
                inherit_acl=True,
            ),
        )
        self.proto.spec.account_id = abc_service

    @property
    def id(self):
        return self.proto.meta.id

    def multi_cluster_deploy_unit(self, *args, **kwargs):
        return MultiClusterDeployUnit(self.registry, self.proto.meta.project_id, self.id, self.proto.spec.deploy_units, *args, **kwargs)


class DeployUnit(object):
    def __init__(self, registry, project_id, stage_id, id_):
        self.registry = registry
        self.project_id = project_id
        self.stage_id = stage_id
        self.id = id_

    def box(self, *args, **kwargs):
        return Box(self.registry, self.stage_id, self.id, self.proto, *args, **kwargs)

    def get_juggler_generator(self, gen):
        deploy_group = "{}@stage={};deploy_unit={}".format(self.project_id, self.stage_id, self.id)
        return self.registry.add_subregistry(gen.clone(
            append_child=False,
            child_group=deploy_group,
            child_group_type=juggler_consts.GroupType.deploy,
            host="{}.{}.{}".format(self.project_id, self.stage_id, self.id),
        ))


class MultiClusterDeployUnit(DeployUnit):
    def __init__(self, registry, project_id, stage_id, deploy_units_proto, pods_per_cluster, cpu_ms, ram, storage, project_network, network_bandwidth, id_="main"):
        super(MultiClusterDeployUnit, self).__init__(registry, project_id, stage_id, id_)

        self.proto = deploy_units_proto.get_or_create(id_)

        self.set_network(
            project_network
        ).add_disk(
            storage
        ).set_resources(
            cpu_ms,
            ram,
            network_bandwidth,
        )

        for dc, replica_count in six.iteritems(pods_per_cluster):
            self.add_cluster(dc, replica_count)

    def deployment_strategy(self, max_unavailable):
        self.proto.multi_cluster_replica_set.replica_set.deployment_strategy.max_unavailable = max_unavailable
        return self

    def set_network(self, project_network):
        self.proto.network_defaults.network_id = project_network
        return self

    def add_endpoint_set(self, port, id_=None):
        endpoint_set = self.proto.endpoint_sets.add(port=port)
        if id_:
            endpoint_set.id = id_
        return self

    def add_cluster(self, dc, replica_count):
        get_replica_set(self.proto).clusters.add(
            cluster=dc,
            spec=data_model.TMultiClusterReplicaSetSpec.TReplicaSetSpecPreferences(
                replica_count=replica_count,
            )
        )
        return self

    def add_disk(self, storage, id_="disk-0"):
        disk = get_pod_template_spec(self.proto).disk_volume_requests.add(
            id=id_,
            quota_policy=storage.proto,
            storage_class=storage.type,
        )
        disk.labels.attributes.add(
            key="used_by_infra",
            value="%true",
        )
        return self

    def set_resources(self, cpu_ms, ram, network_bandwidth):
        get_pod_template_spec(self.proto).resource_requests.CopyFrom(data_model.TPodSpec.TResourceRequests(
            vcpu_guarantee=cpu_ms,
            vcpu_limit=cpu_ms,
            memory_guarantee=ram.total_bytes(),
            memory_limit=ram.total_bytes(),
            network_bandwidth_guarantee=network_bandwidth.total_bytes(),
        ))
        return self

    def add_balancer(self, balancer):
        if not self.proto.network_defaults.network_id:
            raise Exception("Network not set")

        get_pod_template_spec(self.proto).ip6_address_requests.add(
            network_id=self.proto.network_defaults.network_id,
            vlan_id="backbone",
            enable_dns=True,
            virtual_service_ids=[balancer]
        )
        return self


def get_replica_set(deploy_unit):
    return deploy_unit.multi_cluster_replica_set.replica_set


def get_pod_template_spec(deploy_unit):
    return get_replica_set(deploy_unit).pod_template_spec.spec


def get_pod_agent_spec(deploy_unit):
    return get_pod_template_spec(deploy_unit).pod_agent_payload.spec


def get_secrets(deploy_unit):
    return get_pod_template_spec(deploy_unit).secret_refs


def add_secret(deploy_unit, secret_env):
    alias = "{}:{}".format(secret_env.sec, secret_env.ver)

    secret = get_secrets(deploy_unit).get_or_create(alias)
    secret.secret_id = secret_env.sec
    secret.secret_version = secret_env.ver

    return alias


class Storage(object):
    def __init__(self, capacity, bandwidth_guarantee, bandwidth_limit=None):
        self.proto = data_model.TPodSpec.TDiskVolumeRequest.TQuotaPolicy(
            capacity=capacity.total_bytes(),
            bandwidth_guarantee=bandwidth_guarantee.total_bytes(),
            bandwidth_limit=(bandwidth_limit or bandwidth_guarantee).total_bytes(),
        )


class HDD(Storage):
    type = "hdd"


class SSD(Storage):
    type = "ssd"


class Box(object):
    def __init__(self, registry, stage_id, deploy_unit_id, deploy_unit_proto, docker_image, id_="main", docker_image_tag=None):
        self.registry = registry
        self.stage_id = stage_id
        self.deploy_unit_id = deploy_unit_id

        self.deploy_unit_proto = deploy_unit_proto
        self.proto = get_pod_agent_spec(deploy_unit_proto).boxes.add(
            id=id_,
        )
        self.set_image_proto(id_, docker_image, docker_image_tag)

    def set_image_proto(self, id_, image, image_tag):
        image_proto = self.deploy_unit_proto.images_for_boxes.get_or_create(id_)
        image_proto.name = image
        image_proto.registry_host = "registry.yandex.net"
        image_proto.digest = "EMPTY"
        if image_tag:
            image_proto.tag = image_tag

        return image_proto

    def add_init_command(self, cmd):
        self.proto.init.extend([cmd.proto])
        return self

    def workload(self, *args, **kwargs):
        return Workload(self, *args, **kwargs)

    @property
    def id(self):
        return self.proto.id

    def sandbox_release_rule(self, task_type, release_type):
        return self.registry.store(SandboxReleaseRule.TAG, SandboxReleaseRule(self, task_type, release_type))

    def docker_release_rule(self, release_type):
        self.registry.store(DockerReleaseRule.TAG, DockerReleaseRule(self, release_type))
        return self

    def literal_env(self, literal_env):
        env = self.proto.env.add(name=literal_env.env)
        env.value.literal_env.value = literal_env.value
        return self

    def secret_env(self, secret_env):
        env = self.proto.env.add(name=secret_env.env)
        env.value.secret_env.alias = add_secret(self.deploy_unit_proto, secret_env)
        env.value.secret_env.id = secret_env.id
        return self


class Workload(object):
    def __init__(self, box, start_cmd=None, id_="main"):
        self.deploy_unit_proto = box.deploy_unit_proto

        self.proto = get_pod_agent_spec(self.deploy_unit_proto).workloads.add(
            box_ref=box.proto.id,
            id=id_,
            transmit_logs=True,
        )

        if start_cmd:
            self.proto.start.command_line = start_cmd

        get_pod_agent_spec(self.deploy_unit_proto).mutable_workloads.add(
            workload_ref=id_,
        )

    def liveness_http_check(self, path, port=80):
        self.proto.liveness_check.http_get.CopyFrom(data_model.THttpGet(
            path=path,
            port=port,
            any=True,
        ))
        return self

    def liveness_cmd_check(self, cmd):
        self.proto.liveness_check.container.CopyFrom(cmd.proto)
        return self

    def readiness_http_check(self, path, port=80):
        self.proto.readiness_check.http_get.CopyFrom(data_model.THttpGet(
            path=path,
            port=port,
            any=True,
        ))
        return self

    def readiness_cmd_check(self, cmd):
        self.proto.readiness_check.container.CopyFrom(cmd.proto)
        return self

    def literal_env(self, literal_env):
        env = self.proto.env.add(name=literal_env.env)
        env.value.literal_env.value = literal_env.value
        return self

    def secret_env(self, secret_env):
        env = self.proto.env.add(name=secret_env.env)
        env.value.secret_env.alias = add_secret(self.deploy_unit_proto, secret_env)
        env.value.secret_env.id = secret_env.id
        return self

    def add_coredumps(self, count_limit=3, probability=100, total_size_limit_megabytes=1024, cleanup_ttl=datetime.timedelta(hours=1), aggregate=True):
        config = self.deploy_unit_proto.coredump_config[self.proto.id].coredump_processor
        config.count_limit = count_limit
        config.probability = probability
        config.total_size_limit_megabytes = total_size_limit_megabytes
        config.cleanup_ttl_seconds = int(cleanup_ttl.total_seconds())
        config.aggregator.enabled = aggregate
        return self


class CryptaDeployStageGenerator(DeployStageGenerator):
    def __init__(self):
        super(CryptaDeployStageGenerator, self).__init__(abc_service=consts.ABC_CRYPTA_DEV, project="crypta")


class ReleaseRule(object):
    TAG = "deploy.release_rule"

    def __init__(self, box, id_):
        self.box = box

        self.proto = data_model.TReleaseRule(
            meta=data_model.TReleaseRuleMeta(
                id=id_[:consts.RELEASE_RULE_ID_MAX_LENGTH],
                stage_id=box.stage_id,
                inherit_acl=True,
            ),
        )
        self.proto.spec.auto_commit_policy.type = data_model.TAutoCommitPolicy.EType.MAINTAIN_ACTIVE_TRUNK


class DockerReleaseRule(ReleaseRule):
    def __init__(self, box, release_type):
        super(DockerReleaseRule, self).__init__(box, "{}-{}-{}".format(box.stage_id, box.deploy_unit_id, box.id))

        self.proto.spec.docker.image_name = box.deploy_unit_proto.images_for_boxes[box.id].name
        self.proto.spec.docker.release_types.append(release_type)

        patch = self.proto.spec.patches["{}-{}".format(box.deploy_unit_id, box.id)]
        patch.docker.docker_image_ref.box_id = box.id
        patch.docker.docker_image_ref.deploy_unit_id = box.deploy_unit_id

    def fill_urls(self, client):
        pass


class SandboxReleaseRule(ReleaseRule):
    def __init__(self, box, task_class, release_type):
        task_type = str(task_class)
        super(SandboxReleaseRule, self).__init__(box, "{}-{}-{}".format(box.stage_id, box.deploy_unit_id, task_type))

        self.proto.spec.sandbox.task_type = task_type
        self.proto.spec.sandbox.release_types.append(release_type)

        self.resource_types = {}

    def static_resource(self, resource_class, resource_id, mount_point):
        resource_type = str(resource_class)
        self.proto.spec.sandbox.resource_types.append(resource_type)

        patch = self.proto.spec.patches["{}-{}".format(self.box.id, resource_id)]
        patch.sandbox.sandbox_resource_type = resource_type
        patch.sandbox.static.deploy_unit_id = self.box.deploy_unit_id
        patch.sandbox.static.static_resource_ref = resource_id

        static_resource = get_pod_agent_spec(self.box.deploy_unit_proto).resources.static_resources.add(
            id=resource_id,
        )
        static_resource.verification.checksum = "EMPTY:"

        self.box.proto.static_resources.add(
            resource_ref=resource_id,
            mount_point=mount_point,
        )

        self.resource_types[resource_id] = resource_type

    def fill_urls(self, client):
        for static_resource in get_pod_agent_spec(self.box.deploy_unit_proto).resources.static_resources:
            static_resource.url = "sbr:{}".format(client.get_sandbox_resource_id(self.resource_types[static_resource.id], self.proto.spec.sandbox.release_types[0]))


def dt_to_ms(dt):
    return int(dt.total_seconds() * 1000)


class Command(object):
    def __init__(self, cmd):
        self.proto = data_model.TUtilityContainer(command_line=cmd)

    def initial_delay(self, dt):
        self.proto.time_limit.initial_delay_ms = dt_to_ms(dt)
        return self

    def min_restart_period(self, dt):
        self.proto.time_limit.min_restart_period_ms = dt_to_ms(dt)
        return self

    def max_restart_period(self, dt):
        self.proto.time_limit.max_restart_period_ms = dt_to_ms(dt)
        return self

    def max_execution_time(self, dt):
        self.proto.time_limit.max_execution_time_ms = dt_to_ms(dt)
        return self
