from __future__ import absolute_import
import re

import juggler_sdk
import six

from crypta.lib.python.spine.juggler import consts


class JugglerAggregateCheck(juggler_sdk.Check):
    """
    juggler_sdk.Check wrapper with convenient methods for check modification
    All methods return self
    https://docs.yandex-team.ru/juggler/aggregates/basics
    """
    REGISTRY_TAG = "juggler_aggregate_check"

    def __init__(self, escalation_tag, *args, **kwargs):
        """
        :param escalation_tag: tag that is manipulated with *phone_escalation methods, a notify rule for this tag
                               with phone escalation is supposed to exist in Juggler
        """
        kwargs.setdefault("aggregator_kwargs", {})
        super(JugglerAggregateCheck, self).__init__(*args, **kwargs)
        self.escalation_tag = escalation_tag
        self.ignore_unreachable = False
        self.set_host(self.host)

    def add_yt_dependencies(self, proxies):
        """
        Disable this check when YT clusters are in maintenance
        :param proxies: sequence of YT proxy FQDNs
        :return:
        """
        if proxies:
            self.aggregator_kwargs.setdefault("unreach_checks", []).extend([("{}:ongoing_maintenance".format(proxy)) for proxy in proxies])
        return self

    def add_phone_escalation(self):
        assert self.escalation_tag is not None
        if self.escalation_tag not in self.tags:
            self.add_tag(self.escalation_tag)
        return self

    def remove_phone_escalation(self):
        self.tags = [tag for tag in self.tags if tag != self.escalation_tag]
        return self

    def set_phone_escalation(self, value):
        return self.add_phone_escalation() if value else self.remove_phone_escalation()

    def add_nodata_mode(self, mode):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#no-data
        :param mode: :class:`~crypta.lib.python.spine.juggler.consts.NoDataMode`
        """
        self.aggregator_kwargs.update(nodata_mode=mode)
        return self

    def set_nodata_desc(self, desc):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#forced-description
        """
        self.aggregator_kwargs.update(nodata_desc=desc)
        return self

    def skip_unreachable(self, services):
        if not self.ignore_unreachable:
            if not services:
                return self

            self.set_shared_skip_unreachable([{"check": service} for service in services])

        return self

    def set_shared_skip_unreachable(self, shared_unreach_service_kwarg=None, mode="skip"):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#unreach-service
        :param shared_unreach_service_kwarg: [{"check": service}...], is not copied but shared
        :param mode: :class:`~crypta.lib.python.spine.juggler.consts.UnreachMode`
        """
        if shared_unreach_service_kwarg is not None:
            self.aggregator_kwargs.update(
                unreach_service=shared_unreach_service_kwarg,
                unreach_mode=mode,
            )

        return self

    def reset_unreachable(self):
        self.aggregator_kwargs.pop("unreach_service")
        self.aggregator_kwargs.pop("unreach_mode")
        return self

    def add_flap_detector(self, params):
        """
        https://docs.yandex-team.ru/juggler/aggregates/flaps
        :param params: :class:`~crypta.lib.python.spine.juggler.flap_detector_params.FlapDetectorParams`
        """
        self.flaps_config = juggler_sdk.FlapOptions(
            stable=params.stable_time.total_seconds(),
            critical=params.crit_time.total_seconds(),
            boost=params.boost_time.total_seconds() if params.boost_time else 0,
        )

        return self

    def set_multiple_children(self, child_hosts=None, child_services=None, group_type=consts.GroupType.host):
        """
        Aggregate child_hosts x child_services
        https://docs.yandex-team.ru/juggler/aggregates/basics#children
        """
        self.children = []
        for child_host in child_hosts or [self.host]:
            for child_service in child_services or [self.service]:
                self.add_children(child_host, child_service, group_type)
        return self

    def add_children(self, group, service=None, group_type=consts.GroupType.host):
        """
        https://docs.yandex-team.ru/juggler/aggregates/basics#children
        """
        self.children.append(self._create_child(group, service, group_type))
        return self

    def set_child(self, group, service=None, group_type=consts.GroupType.host, instance=None):
        """
        https://docs.yandex-team.ru/juggler/aggregates/basics#children
        """
        self.children = [self._create_child(group, service, group_type, instance)]
        return self

    def _create_child(self, group, service, group_type, instance=None):
        if group_type == consts.GroupType.events:
            # https://docs.yandex-team.ru/juggler/aggregates/groups#events
            assert service is None or service == consts.ALL, "service must be None or 'all' for EVENTS child"
            return juggler_sdk.Child(
                host=group,
                service=consts.ALL,
                group_type=group_type,
                instance=consts.ALL,
            )

        return juggler_sdk.Child(
            host=group,
            service=service or self.service,
            group_type=group_type,
            instance=instance,
        )

    def add_tag(self, tag):
        self.tags.append(tag)
        return self

    def add_tags(self, tags):
        self.tags += tags or []
        return self

    def make_active(self, name, **kwargs):
        """
        https://docs.yandex-team.ru/juggler/aggregates/actives
        """
        self.active = name
        if kwargs:
            self.active_kwargs = {k: v for k, v in six.iteritems(kwargs) if v is not None}
        return self

    def set_aggregator(self, aggregator, kwargs=None):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators
        """
        self.aggregator = aggregator
        self.aggregator_kwargs.update(kwargs or {})
        return self

    def set_meta(self, meta):
        """
        https://docs.yandex-team.ru/juggler/aggregates/basics
        """
        self.meta = meta
        return self

    def set_crit_limit(self, crit_limit):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#timed_more_than_limit_is_problem
        :param crit_limit: :class:`~datetime.timedelta`
        """
        assert self.aggregator == "timed_more_than_limit_is_problem"
        self.aggregator_kwargs["limits"][0]["crit"] = crit_limit
        return self

    def set_warn_limit(self, warn_limit):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#timed_more_than_limit_is_problem
        :param warn_limit: :class:`~datetime.timedelta`
        """
        assert self.aggregator == "timed_more_than_limit_is_problem"
        self.aggregator_kwargs["limits"][0]["warn"] = warn_limit
        return self

    def set_ttl(self, ttl):
        """
        https://docs.yandex-team.ru/juggler/aggregates/basics
        :param ttl: int seconds
        """
        self.ttl = ttl
        return self

    def set_refresh_time(self, refresh_time):
        """
        https://docs.yandex-team.ru/juggler/aggregates/basics
        :param ttl: int seconds
        """
        self.refresh_time = refresh_time
        return self

    def make_frequent(self):
        """
        Set ttl and refresh_time to predefined frequents constants
        """
        return self.set_ttl(consts.FREQUENT_TTL).set_refresh_time(consts.FREQUENT_REFRESH_TIME)

    def set_host(self, host):
        if host is not None:
            self.host = re.sub(r"(?![\w.-]).", "_", host)
        return self

    @property
    def unreachable_services(self):
        return [x["check"] for x in self.aggregator_kwargs.get("unreach_service", [])]

    def to_dict(self, minimize=False):
        result = super(JugglerAggregateCheck, self).to_dict(minimize=minimize)
        if not self.unreachable_services:
            aggregator_kwargs = result.get("aggregator_kwargs", {})
            aggregator_kwargs.pop("unreach_service", None)
            aggregator_kwargs.pop("unreach_mode", None)
        return result

    def add_url(self, title, url, type_):
        self.meta.setdefault("urls", []).append({
            "title": title,
            "url": url,
            "type": type_,
        })
        return self

    def add_yt_url(self, proxy, path, title="YT"):
        url = "https://yt.yandex-team.ru/{}/navigation?path={}".format(
            proxy.split(".", 1)[0],
            path,
        )
        return self.add_url(title, url, "doc")

    def add_solomon_url(self, alert, title="Solomon"):
        url = "https://solomon.yandex-team.ru/admin/projects/{}/alerts/{}".format(
            alert.projectId,
            alert.id,
        )
        return self.add_url(title, url, "doc")
