from __future__ import absolute_import
import copy
import datetime

import six

from crypta.lib.python.spine.config_registry import ConfigRegistry
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.juggler.juggler_aggregate_check import JugglerAggregateCheck


class JugglerCheckGenerator(ConfigRegistry):
    """
    Juggler aggregate check generator that acts as a ConfigRegistry
    All checks are stored inside generator
    """
    class Config(object):
        """
        Default parameters for aggregate checks
        """
        def __init__(
            self,
            child_group=None,
            child_group_type=consts.GroupType.host,
            host=None,
            warn_limit=0,
            crit_limit=0,
            escalation=False,
            tags=None,
            nodata_desc=None,
            nodata_mode=None,
            ttl=None,
            refresh_time=None,
            yt_dependencies=None,
            escalation_tag=None,
        ):
            self.child_group = child_group
            self.child_group_type = child_group_type
            self.host = host
            self.warn_limit = warn_limit
            self.crit_limit = crit_limit
            self.escalation = escalation
            self.tags = tags or []
            self.nodata_desc = nodata_desc
            self.nodata_mode = nodata_mode
            self.ttl = ttl
            self.refresh_time = refresh_time
            self.yt_dependencies = yt_dependencies
            self.escalation_tag = escalation_tag

    def __init__(self, **kwargs):
        """
        If host is not specified, child_group will be used as host
        """
        super(JugglerCheckGenerator, self).__init__()
        self.default_config = self.Config(**kwargs)
        self.shared_unreach_service_kwarg = []

    def _base(self, service, aggregator=None, aggregator_kwargs=None, is_unreachable=False):
        check = self.store(JugglerAggregateCheck.REGISTRY_TAG, JugglerAggregateCheck(
            escalation_tag=self.default_config.escalation_tag,
            host=self.default_config.host or self.default_config.child_group,
            service=service,
            ttl=self.default_config.ttl or consts.DEFAULT_TTL,
            refresh_time=self.default_config.refresh_time or consts.DEFAULT_REFRESH_TIME,
            tags=list(self.default_config.tags),
            namespace=consts.NAMESPACE,
            mark=consts.MARK,
        ))

        if aggregator:
            check.set_aggregator(aggregator, aggregator_kwargs).set_shared_skip_unreachable(None if is_unreachable else self.shared_unreach_service_kwarg)
            if is_unreachable:
                self.shared_unreach_service_kwarg.append({"check": ":{}".format(check.service)})

        if self.default_config.nodata_mode is not None:
            check.add_nodata_mode(self.default_config.nodata_mode)

        if self.default_config.nodata_desc is not None:
            check.set_nodata_desc(self.default_config.nodata_desc)

        if self.default_config.escalation:
            check.add_phone_escalation()

        if self.default_config.yt_dependencies:
            check.add_yt_dependencies(self.default_config.yt_dependencies)

        return check

    def icmpping(self, is_unreachable=True):
        """
        https://docs.yandex-team.ru/juggler/aggregates/actives#icmpping
        :param is_unreachable: disable other checks when this one is critical
        """
        check = self.some("UNREACHABLE", is_unreachable=is_unreachable).make_active("icmpping").make_frequent()
        if self.default_config.child_group_type == consts.GroupType.qloud:
            check.add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=15), datetime.timedelta(minutes=45)))
        return check

    def ssh(self):
        """
        https://docs.yandex-team.ru/juggler/aggregates/actives#ssh
        """
        return self.some("ssh").make_active("ssh", disable_ipv4="yes").make_frequent()

    def http(self, service, path, port, substr=None, is_unreachable=True, unreach_services=consts.DEFAULT_UNREACH_SERVICES):
        """
        https://docs.yandex-team.ru/juggler/aggregates/actives#http
        :param is_unreachable: disable other checks when this one is critical
        :param unreach_services: list of service that disable this check
        """
        return self.some(service, child_service="http", is_unreachable=is_unreachable).make_active(
            "http",
            path=path,
            port=port,
            always_show_body=True,
            ok_codes=[200],
            substr=substr,
        ).skip_unreachable(unreach_services).make_frequent()

    def tvm_client_status(self):
        """
        Check that C++ TVM client is operational (able to update its state from TVM)
        Signals for this check are generated by TTvmStatusReporter
        https://a.yandex-team.ru/arc/trunk/arcadia/crypta/lib/native/tvm/tvm_status_reporter.cpp?rev=6459238#L5
        """
        return self.some("tvm-client-status").set_crit_limit("100%")

    def some(self, service, is_unreachable=False, child_service=None):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#timed_more_than_limit_is_problem
        :param is_unreachable: disable other checks when this one is critical
        """
        return self._base(
            service,
            aggregator="timed_more_than_limit_is_problem",
            aggregator_kwargs={
                "limits": [{
                    "time_start": 0,
                    "time_end": 23,
                    "day_start": 1,
                    "warn": self.default_config.warn_limit,
                    "day_end": 7,
                    "crit": self.default_config.crit_limit,
                }]
            },
            is_unreachable=is_unreachable
        ).set_child(self.default_config.child_group, child_service or service, self.default_config.child_group_type)

    def graphite_client(self):
        """
        Check that graphite-client is able to push metrics to Graphite
        """
        return self.some("graphite-client") \
            .set_crit_limit(100) \
            .add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=15), datetime.timedelta(minutes=45)))

    def push_client(self, log_file):
        """
        Check that push-client is able to push log files to Logbroker
        """
        return self.some("push_{}".format(log_file))\
            .add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=15), datetime.timedelta(minutes=45)))

    def direct(self, service):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#no-aggregator
        """
        return self._base(service)

    def _logic(self, type, service, child_service):
        return self._base(service, type) \
            .set_child(self.default_config.child_group, child_service or service, self.default_config.child_group_type)

    def any(self, service, child_service=None):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#logic_or
        """
        return self._logic("logic_or", service, child_service)

    def all(self, service, child_service=None):
        """
        https://docs.yandex-team.ru/juggler/aggregates/aggregators#logic_and
        """
        return self._logic("logic_and", service, child_service)

    def lsf(self, service, crit_time, child_host):
        """
        lsf = last successful finish
        Check that last OK was sent at least crit_time ago
        :param crit_time: :class:`~datetime.timedelta`
        """
        return self.any(service)\
            .set_ttl(int(crit_time.total_seconds()))\
            .set_nodata_desc("No OK for more than {}".format(crit_time))\
            .set_child(child_host) \
            .set_shared_skip_unreachable([])

    # TODO(cherenkov-p-a) Maybe put walle to separate module?
    def walle_yasm_alert(self, host, service, yasm_alert_name):
        return self.any(service)\
            .set_refresh_time(5)\
            .set_ttl(900)\
            .remove_phone_escalation()\
            .set_host(host)\
            .set_child("yasm_alert", yasm_alert_name)\
            .add_nodata_mode(consts.NoDataMode.force_warn)\
            .set_shared_skip_unreachable(
                [{"check": "yasm_alert:virtual-meta"}],
                consts.UnreachMode.force_ok
            ).set_meta({
                "yasm_alert_name": yasm_alert_name,
                "urls": [{
                    "url": "https://yasm.yandex-team.ru/chart-alert/alerts={};".format(yasm_alert_name),
                    "type": "yasm_alert",
                }],
            })

    def walle_healing_automation_enabled(self, host, walle_project):
        yasm_alert_name = "{}-wall-e-healing".format(walle_project)
        service = "wall-e-healing"
        return self.walle_yasm_alert(host, service, yasm_alert_name)

    def walle_dns_automation_enabled(self, host, walle_project):
        yasm_alert_name = "{}-wall-e-dns".format(walle_project)
        service = "wall-e-dns"
        return self.walle_yasm_alert(host, service, yasm_alert_name)

    def add_walle_checks(self, host, walle_project):
        self.walle_healing_automation_enabled(host, walle_project),
        self.walle_dns_automation_enabled(host, walle_project),

    def periodic_task(self, juggler_service, crit_time, env=environment.PRODUCTION):
        """
        Check when binary run by periodic_task had latest successful finish
        :param crit_time: :class:`~datetime.timedelta`
        """
        return self.lsf(juggler_service, crit_time, consts.ENV_TO_PERIODIC_TASK_HOST[env])

    def clone(self, append_child=True, **kwargs):
        """
        Create check generator with the same default parameters and save is as subregistry of this generator
        :param append_child: append child to registry
        :param kwargs: overrides for new generator default parameters
        :return: :class:`JugglerCheckGenerator`
        """
        kwargs["tags"] = self.default_config.tags + kwargs.get("tags", [])
        for k, v in six.iteritems(self.default_config.__dict__):
            if k not in kwargs or kwargs[k] is None:
                kwargs[k] = copy.deepcopy(self.default_config.__dict__[k])

        gen = JugglerCheckGenerator(**kwargs)
        if append_child:
            self.add_subregistry(gen)

        return gen


class CryptaYtCheckGenerator(JugglerCheckGenerator):
    def __init__(self, tags=None, **kwargs):
        super(CryptaYtCheckGenerator, self).__init__(escalation_tag="crypta-yt-phone-escalation", tags=["crypta-yt-all"] + (tags or []), **kwargs)
