import collections
import datetime
import enum

from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.config_registry import ConfigRegistry
from crypta.lib.python.spine.solomon import (
    solomon_alert_utils,
    solomon_check_generator,
)


CLUSTERS = ("rtmr-vla", "rtmr-sas")


class ProcessingOperationConfig(object):
    def __init__(self, operation, error_threshold=0, fail_threshold=150):
        self.operation = operation
        self.error_threshold = error_threshold
        self.fail_threshold = fail_threshold


OperationForTotalDeliveryTimeConfig = collections.namedtuple("OperationForTotalDeliveryTimeConfig", ["operation", "threshold_seconds"])
TableSizeConfig = collections.namedtuple("TableSizeConfig", ["table", "threshold"])
StateSizeConfig = collections.namedtuple("StateSizeConfig", ["operation", "threshold"])
OutputOperationConfig = collections.namedtuple("OutputOperationConfig", ["operation", "escalation"])


class TotalDeliveryTime(enum.Enum):
    SECONDS_1 = 1
    SECONDS_3 = 3
    SECONDS_5 = 5
    SECONDS_7 = 7
    SECONDS_10 = 10
    SECONDS_15 = 15
    SECONDS_30 = 30
    SECONDS_45 = 45
    SECONDS_60 = 60
    SECONDS_120 = 120
    SECONDS_180 = 180

    @staticmethod
    def get_total_sensor():
        return "TotalDeliveryTime*s"

    @classmethod
    def get_sensor(cls, threshold):
        return "|".join(
            "TotalDeliveryTime{}s".format(i)
            for i in sorted(x.value for x in list(cls))
            if i <= cls(threshold).value
        )


class RtmrCheckGenerator(ConfigRegistry):
    def __init__(self, juggler_check_generator, task_name, project_id, solomon_service, cluster):
        super(RtmrCheckGenerator, self).__init__()
        juggler_check_generator.add_subregistry(self)

        self.task_name = task_name
        self.cluster = cluster
        self.juggler_check_generator = juggler_check_generator

        self.host = "rtmr.{}.{}".format(self.task_name, self.cluster.replace('_', '-'))
        self.raw_host = "{}.raw".format(self.host)
        self.cgroup = cluster.replace("-", "_")

        self.solomon_check_generator = solomon_check_generator.SolomonCheckGenerator(
            juggler_check_generator.clone(
                warn_limit=0,
                crit_limit=100,
                child_group_type=consts.GroupType.host,
                child_group=self.raw_host,
                host=self.host,
            ),
            solomon_alert_utils.AlertCreator(project_id, {
                "project": project_id,
                "service": solomon_service,
                "cluster": self.cluster,
            }),
        )

    def _metrics_tasks_selector(self, app, subcomponent, sensor):
        return {
            "project": "rtmr",
            "service": "metrics_tasks",
            "app": app,
            "subcomponent": subcomponent,
            "component": "Task",
            "host": "cluster",
            "sensor": sensor,
            "cluster": self.cluster,
        }

    def _operation_subcomponent(self, operation):
        return "{}_{}".format(self.task_name, operation)

    def _size_check(self, table, threshold, juggler_service):
        solomon_sensor = self.solomon_check_generator.get_sensor({
            "project": "rtmr",
            "service": "storage",
            "host": "cluster",
            "table": table,
            "sensor": "BytesCount",
            "cluster": self.cluster,
        })
        return solomon_sensor.create_threshold_check(
            aggregation=alert_pb2.AVG,
            predicate=alert_pb2.LT,
            threshold=threshold.total_bytes(),
            period=datetime.timedelta(hours=3),
            description="bytes: {{pointValue}}",
            juggler_service=juggler_service,
        ).set_crit_limit(0)

    def table_size_check(self, config):
        return self._size_check(
            config.table,
            config.threshold,
            "table_size.{}".format(config.table),
        )

    def state_size_check(self, config):
        return self._size_check(
            "_STATE_DATA_{}:{}".format(self.task_name, config.operation),
            config.threshold,
            "state_size.{}".format(config.operation),
        )

    def output_alert(self, operation):
        service = "output.{}".format(operation)

        return self.solomon_check_generator.get_sensor(
            self._metrics_tasks_selector("host", operation, "In")
        ).create_threshold_alert(
            aggregation=alert_pb2.SUM,
            predicate=alert_pb2.LTE,
            threshold=0,
            period=datetime.timedelta(minutes=10),
            description="rows: {{pointValue}}",
            juggler_service=service,
        )

    def error_check(self, config):
        return self.solomon_check_generator.get_sensor({
            "error_table": config.operation,
        }).create_threshold_check(
            aggregation=alert_pb2.SUM,
            predicate=alert_pb2.GT,
            threshold=config.error_threshold,
            period=datetime.timedelta(minutes=10),
            description="rows: {{pointValue}}",
            juggler_service="errors.{}".format(config.operation),
        ).add_nodata_mode(consts.NoDataMode.force_ok)

    def discards_check(self, config):
        return self.solomon_check_generator.get_sensor(
            self._metrics_tasks_selector("server", self._operation_subcomponent(config.operation), "InputEventsDiscarded")
        ).create_threshold_check(
            aggregation=alert_pb2.SUM,
            predicate=alert_pb2.GT,
            threshold=0,
            period=datetime.timedelta(minutes=10),
            description="rows: {{pointValue}}",
            juggler_service="discards.{}".format(config.operation),
        ).add_flap_detector(FlapDetectorParams(
            datetime.timedelta(hours=1),
            datetime.timedelta(hours=3),
        ))

    def fails_check(self, config):
        return self.solomon_check_generator.get_sensor(
            self._metrics_tasks_selector("server", self._operation_subcomponent(config.operation), "Failed")
        ).create_threshold_check(
            aggregation=alert_pb2.AVG,
            predicate=alert_pb2.GT,
            threshold=config.fail_threshold,
            period=datetime.timedelta(minutes=10),
            description="fails: {{pointValue}}",
            juggler_service="fails.{}".format(config.operation),
        ).add_flap_detector(FlapDetectorParams(
            datetime.timedelta(hours=1),
            datetime.timedelta(hours=3),
        ))

    def rtmr_fails_check(self, threshold=0.5):
        return self.solomon_check_generator.create_expression_alert_check(
            service="rtmr_finished_jobs",
            period=datetime.timedelta(minutes=10),
            program="""
            let finishReasons = {{group="finishReason", service="yf functions",
                job="Total", actor="JobMaster", project="yf", cluster="{cluster}",
                host="cluster", version="Total", jobSetPath="crypta",
                sensor="SIGABORTED|SIGNALLED_OOM|SIGSEGVED|SIGNALLED"}};

            let sumFinished = sum(group_lines("sum", finishReasons));
            let threshold = {threshold};

            alarm_if(sumFinished > threshold);
            """.format(cluster=self.cluster, threshold=threshold),
            description="rtmr tasks finished by signal: {{ expression.sumFinished }} > {{ expression.threshold }}",
        )

    def total_delivery_time_check(self, config):
        threshold_seconds = TotalDeliveryTime(config.threshold_seconds)
        service = "total_delivery_time.{}".format(config.operation)
        subcomponent = self._operation_subcomponent(config.operation)

        return self.solomon_check_generator.create_hist_check(
            service=service,
            period=datetime.timedelta(minutes=30),
            labels_all=self._metrics_tasks_selector("server", subcomponent, TotalDeliveryTime.get_total_sensor()),
            labels_bins=self._metrics_tasks_selector("server", subcomponent, TotalDeliveryTime.get_sensor(threshold_seconds)),
            threshold=0.9,
            predicate=alert_pb2.LT,
            description="{} second avg percentile : {{{{expression.avg_percentile}}}} < {{{{expression.threshold}}}}".format(threshold_seconds.value),
        ).add_flap_detector(FlapDetectorParams(
            datetime.timedelta(hours=3),
            datetime.timedelta(hours=9),
        ))

    # TODO(cherenkov-p-a) this does not work anymore because of YF
    def resource_check(self, resource):
        return self.juggler_check_generator.any(resource)\
            .set_host(self.host)\
            .set_child(self.cgroup, group_type=consts.GroupType.conductor)\
            .add_nodata_mode(consts.NoDataMode.skip)\
            .add_flap_detector(FlapDetectorParams(
                datetime.timedelta(minutes=15),
                datetime.timedelta(minutes=45),
            ))


def add_output_check(juggler, common_host, rtmrs, config):
    alerts = [rtmr.output_alert(config.operation) for rtmr in rtmrs]
    service = alerts[0].annotations["juggler-service"]

    check = juggler.all(service)\
        .set_host(common_host)\
        .set_multiple_children(
            (alert.annotations["juggler-host"] for alert in alerts),
            [service],
        )

    if config.escalation:
        check.add_phone_escalation()

    return check


def create_checks(
    juggler_check_generator,
    task_name,
    project_id,
    solomon_service,
    output_operations,
    processing_operations,
    operations_for_total_delivery_time,
    resources,
    tracked_state_sizes,
    tracked_table_sizes,
    clusters=CLUSTERS,
):
    """
    Create juggler aggregate checks associated with RTMR task
    :param juggler_check_generator: :class:`~crypta.lib.python.spine.juggler.juggler_check_generator.JugglerCheckGenerator`
    :param task_name: RTMR task name
    :param project_id: Solomon project
    :param solomon_service: Solomon service label
    :param output_operations: list of :class:`OutputOperationConfig`
    :param processing_operations: list of :class:`ProcessingOperationConfig`
    :param operations_for_total_delivery_time: list of :class:`OperationForTotalDeliveryTimeConfig`
    :param resources: list of juggler services, indicating that resource is parsed correctly
    :param tracked_state_sizes: list of :class:`StateSizeConfig`
    :param tracked_table_sizes: list of :class:`TableSizeConfig`
    :param clusters: RTMR cluster names
    :return: None
    """
    rtmrs = [RtmrCheckGenerator(juggler_check_generator, task_name, project_id, solomon_service, cluster) for cluster in clusters]

    for rtmr in rtmrs:
        for config in processing_operations:
            rtmr.error_check(config)
            rtmr.discards_check(config)
            rtmr.fails_check(config)

        for config in operations_for_total_delivery_time:
            rtmr.total_delivery_time_check(config)

        for resource in resources:
            rtmr.resource_check(resource)

        for config in tracked_state_sizes:
            rtmr.state_size_check(config)

        for config in tracked_table_sizes:
            rtmr.table_size_check(config)

        rtmr.rtmr_fails_check(threshold=0.5)

    for config in output_operations:
        add_output_check(juggler_check_generator, "rtmr.{}".format(task_name), rtmrs, config)
