import six

from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.sandbox import consts
from sandbox.projects.crypta import run_universal_bundle


CUSTOM_FIELDS = "custom_fields"


class SandboxScheduler(dict):
    """
    Sandbox scheduler config
    """
    REGISTRY_TAG = "sandbox_scheduler"

    def __init__(self, juggler_check_generator, config, task_cls, env, start_immediately, scheduler_id):
        """
        :param juggler_check_generator: :class:`~crypta.lib.python.spine.juggler.juggler_check_generator.JugglerCheckGenerator`
        :param config: dict with scheduler configuration
        :param task_cls: class derived from :class:`~sandbox.projects.crypta.common.CryptaTask`
        :param start_immediately: if False, scheduler will not be started when initially created
        :param scheduler_id: if is not None, this config can only track specific prexisiting scheduler
        """
        super(SandboxScheduler, self).__init__(config)
        self.juggler_check_generator = juggler_check_generator
        self.task_cls = task_cls
        self.env = env
        self.start_immediately = start_immediately
        self.scheduler_id = scheduler_id

    def check(self, crit_time, juggler_service=None):
        """
        Create juggler aggregate check for this scheduler
        :param crit_time: :class:`~datetime.timedelta`
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """
        return self.juggler_check_generator.lsf(
            juggler_service or self.task_cls.get_class_juggler_service(),
            crit_time,
            consts.ENV_TO_HOST[self.env],
        ).add_url(
            title="Sandbox",
            type_="doc",
            url="https://sandbox.yandex-team.ru/schedulers?task_type={task_type}&tags={tags}&all_tags=true".format(
                task_type=str(self.task_cls),
                tags=",".join(self["data"]["task"]["tags"]),
            )
        )


class RunUniversalBundleScheduler(SandboxScheduler):
    def check(self, crit_time):
        service = next(x["value"] for x in self["data"]["task"][CUSTOM_FIELDS] if x["name"] == "juggler_service")
        return super(RunUniversalBundleScheduler, self).check(crit_time, service)


class SandboxSchedulerGenerator(object):
    def __init__(self, juggler_check_generator, author, owner, tags=None):
        """
        :param juggler_check_generator: :class:`~crypta.lib.python.spine.juggler.juggler_check_generator.JugglerCheckGenerator`
        :param tags: list of Sandbox tags
        """
        self.juggler_check_generator = juggler_check_generator
        self.author = author
        self.owner = owner
        self.tags = tags

    def create_scheduler(
        self,
        task_cls,
        schedule_interval=None,
        schedule_daily_start_time=None,
        env=environment.STABLE,
        tags=None,
        kill_timeout=None,
        extra_params=None,
        env_as_param=True,
        requirements=None,
        notifications=None,
        retry_interval=None,
        start_immediately=True,
        scheduler_id=None,
        description=None,
        sequential_run=True,
        scheduler_class=SandboxScheduler,
    ):
        """
        Create scheduler config and adds it as a subregistry of self.juggler_check_generator
        :param task_cls: class derived from :class:`~sandbox.projects.crypta.common.CryptaTask`
        :param schedule_interval: :class:`~datetime.timedelta`
        :param schedule_daily_start_time: isoformat string
        :param tags: list of Sandbox tags
        :param kill_timeout: :class:`~datetime.timedelta`
        :param extra_params: task specific parameters
        :param env_as_param: if True, adds env as "environment" parameter
        :param retry_interval: :class:`~datetime.timedelta`
        :param start_immediately: if False, scheduler will not be started when initially created
        :param scheduler_id: if is not None, this config can only track specific prexisiting scheduler
        :return: :class:`SandboxScheduler`
        """
        assert schedule_interval is None or schedule_daily_start_time is None

        if env == environment.PRODUCTION:
            env = environment.STABLE

        params = dict(extra_params or {})
        if env_as_param:
            params["environment"] = env

        tags = (tags or []) + self.tags
        if env in consts.ENV_TO_TAG:
            tags.append(consts.ENV_TO_TAG[env])

        if scheduler_id is not None:
            tags.append(consts.FIXED_ID)

        config = {
            "task_type": str(task_cls),
            "data": {
                "owner": self.owner,
                "author": self.author,
                "schedule": {
                    "sequential_run": sequential_run,
                },
                "task": {
                    "owner": self.owner,
                    "priority": {
                        "class": "SERVICE",
                        "subclass": "HIGH",
                    },
                    "description": generate_description_with_id(params, description) if scheduler_id is None else description,
                    "tags": [tag.upper() for tag in tags],
                    CUSTOM_FIELDS: [
                        {"name": name, "value": value}
                        for name, value in six.iteritems(params)
                    ],
                    "requirements": requirements or {},
                    "notifications": notifications or [],
                },
            },
        }
        if kill_timeout:
            config["data"]["task"]["kill_timeout"] = kill_timeout.total_seconds()
        if retry_interval:
            config["data"]["schedule"].setdefault("retry", {})["interval"] = retry_interval.total_seconds()
        if schedule_interval:
            config["data"]["schedule"].setdefault("repetition", {})["interval"] = schedule_interval.total_seconds()
        else:
            config["data"]["schedule"].setdefault("repetition", {})["weekly"] = [0, 1, 2, 3, 4, 5, 6]
            config["data"]["schedule"]["start_time"] = schedule_daily_start_time

        scheduler = scheduler_class(self.juggler_check_generator, config, task_cls, env, start_immediately, scheduler_id)

        return self.juggler_check_generator.store(scheduler.REGISTRY_TAG, scheduler)

    def create_run_universal_bundle_scheduler(
        self,
        bundle_name,
        cmd,
        env=environment.STABLE,
        additional_env=None,
        secrets_env=None,
        templates=None,
        juggler_service=None,
        semaphore_name=None,
        semaphore_suffix=None,
        task_name=None,
        output_resources=None,
        use_semaphore=True,
        template_rendering_context=None,
        **kwargs
    ):
        output_resources = output_resources or []

        if use_semaphore:
            semaphore_name = semaphore_name or run_universal_bundle.CryptaRunUniversalBundle._join_not_empty(
                run_universal_bundle.CryptaRunUniversalBundle.__name__, bundle_name, task_name, env, semaphore_suffix,
            )
        else:
            semaphore_name = None

        kwargs.setdefault("extra_params", {}).update({
            "bundle_name": bundle_name,
            "cmd": cmd,
            "additional_env": additional_env or {},
            "secrets_env": secrets_env or {},
            "templates": templates if templates is not None else ["config.yaml"],
            "juggler_service": juggler_service or run_universal_bundle.CryptaRunUniversalBundle._join_not_empty(bundle_name, task_name),
            "semaphore_name": semaphore_name,
            "task_name": task_name,
            "output_resources": [x.to_dict() for x in output_resources],
            "template_rendering_context": template_rendering_context or {},
        })

        kwargs.setdefault("tags", []).append(bundle_name)
        kwargs.setdefault("description", "{}[{}]: {}".format(bundle_name, env, " ".join(cmd)))

        return self.create_scheduler(
            run_universal_bundle.CryptaRunUniversalBundle,
            env=env,
            scheduler_class=RunUniversalBundleScheduler,
            **kwargs
        )


def generate_description_with_id(params, description):
    description = "" if description is None else description + "\n"
    description += " ".join(str(value) for key, value in sorted(six.iteritems(params)))
    return description


def get_id_from_description(description):
    return description.rsplit("\n", 1)[-1]


def create_default_generator(juggler_check_generator, tags=None):
    return SandboxSchedulerGenerator(juggler_check_generator, "robot-crypta", "CRYPTA", tags)


class NotificationStatus(object):
    BREAK = "BREAK"
    NO_RES = "NO_RES"
    EXCEPTION = "EXCEPTION"
    STOPPED = "STOPPED"
    TIMEOUT = "TIMEOUT"
    FAILURE = "FAILURE"
    FINISH = "FINISH"
    EXPIRED = "EXPIRED"
    EXECUTE = "EXECUTE"


def create_email_notifications(logins, statuses):
    return [{
        "transport": "email",
        "recipients": list(logins),
        "statuses": list(statuses),
    }]
