import base64
import hashlib

from crypta.lib.python.solomon.proto import alert_pb2


PREDICATE_TO_STRING = {
    alert_pb2.EQ: "==",
    alert_pb2.NE: "!=",
    alert_pb2.GT: ">",
    alert_pb2.LT: "<",
    alert_pb2.GTE: ">=",
    alert_pb2.LTE: "<=",
}


def serialize_selectors(selectors, exclude_selectors=None):
    exclude_selectors = exclude_selectors or {}
    return "{{{}}}".format(", ".join(
        ["{}='{}'".format(k, v) for k, v in selectors.iteritems()] +
        ["{}!='{}'".format(k, v) for k, v in exclude_selectors.iteritems()]
    ))


def create_base_alert(name, project_id, period, juggler_description,
                      notification_channels=("crypta-juggler",), group_by_labels=("host",), delay=0,
                      juggler_host="{{ labels.host }}", juggler_service=None):
    alert = alert_pb2.TAlert()

    alert.id = base64.b64encode(hashlib.sha384(name).digest(), altchars=b"-_")
    alert.name = name
    alert.projectId = project_id
    alert.periodMillis = int(period.total_seconds() * 1000)

    alert.delaySeconds = delay

    if notification_channels is not None:
        alert.notificationChannels.extend(notification_channels)

    if group_by_labels is not None:
        alert.groupByLabels.extend(group_by_labels)

    alert.annotations["description"] = juggler_description
    alert.annotations["juggler-host"] = juggler_host
    alert.annotations["juggler-service"] = juggler_service or alert.name

    return alert


def create_threshold_alert(name, project_id, time_aggregation, predicate, threshold, period, selectors, juggler_description,
                           notification_channels=("crypta-juggler",), group_by_labels=("host",), delay=0,
                           juggler_host="{{ labels.host }}", juggler_service=None, exclude_selectors=None):
    alert = create_base_alert(
        name=name,
        project_id=project_id,
        period=period,
        juggler_description=juggler_description,
        notification_channels=notification_channels,
        group_by_labels=group_by_labels,
        delay=delay,
        juggler_host=juggler_host,
        juggler_service=juggler_service,
    )

    alert.type.threshold.selectors = serialize_selectors(selectors, exclude_selectors)
    alert.type.threshold.timeAggregation = time_aggregation
    alert.type.threshold.predicate = predicate
    alert.type.threshold.threshold = threshold

    predicate_rule = alert.type.threshold.predicateRules.add()
    predicate_rule.thresholdType = time_aggregation
    predicate_rule.comparison = predicate
    predicate_rule.threshold = threshold
    predicate_rule.targetStatus = alert_pb2.ALARM

    return alert


def create_expression_alert(name, project_id, period, program, juggler_description, check_expression="",
                            notification_channels=("crypta-juggler",), group_by_labels=("host",), delay=0,
                            juggler_host="{{ labels.host }}", juggler_service=None):
    alert = create_base_alert(
        name=name,
        project_id=project_id,
        period=period,
        juggler_description=juggler_description,
        notification_channels=notification_channels,
        group_by_labels=group_by_labels,
        delay=delay,
        juggler_host=juggler_host,
        juggler_service=juggler_service,
    )

    alert.type.expression.program = program
    alert.type.expression.checkExpression = check_expression

    return alert


class AlertCreator(object):
    def __init__(self, project_id, selectors=None):
        self.project_id = project_id
        self.selectors = selectors or {}

    def create_threshold_alert(self, selectors=None, **kwargs):
        result_selectors = {}
        result_selectors.update(self.selectors)
        result_selectors.update(selectors or {})

        return create_threshold_alert(project_id=self.project_id, selectors=result_selectors, **kwargs)

    def create_expression_alert(self, **kwargs):
        return create_expression_alert(project_id=self.project_id, **kwargs)
