import six

import juggler_sdk

from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.solomon import solomon_alert_utils
from crypta.lib.python.spine.solomon.solomon_alert_registry import SolomonAlertRegistry
from crypta.lib.python.spine.solomon.solomon_sensor import SolomonSensor


class SolomonCheckGenerator(SolomonAlertRegistry):
    """
    Creates Solomon alerts and corresponding Juggler aggregate checks
    Solomon project needs to have a Juggler channel
    Example: https://solomon.yandex-team.ru/admin/projects/crypta_stats/channels/crypta-juggler
    """
    def __init__(
        self,
        juggler_check_generator,
        alert_creator,
    ):
        """
        :param juggler_check_generator: :class:`~crypta.lib.python.spine.juggler.juggler_check_generator.JugglerCheckGenerator`
        :param alert_creator: :class:`~crypta.lib.python.spine.solomon.solomon_alert_utils.AlertCreator`
        """
        super(SolomonCheckGenerator, self).__init__()
        self.juggler_check_generator = juggler_check_generator
        self.juggler_check_generator.add_subregistry(self)
        self.alert_creator = alert_creator

    def get_sensor(self, sensor):
        """
        Get a solomon sensor representation which can be used to create threshold alerts https://wiki.yandex-team.ru/solomon/userguide/alerting/#threshold
        :param sensor: if string, will be added as value for label "sensor", otherwise should be a dict of labels
        :return: :class:`~crypta.lib.python.spine.solomon.solomon_sensor.SolomonSensor`
        """
        return self.add_subregistry(SolomonSensor(
            self.juggler_check_generator,
            self.alert_creator,
            sensor,
        ))

    def _create_expression_alert_check(
        self,
        service,
        period,
        program,
        description,
        solomon_juggler_host,
        group_by_labels,
        juggler_children,
    ):
        """
        Create Solomon expression alert and corresponding Juggler aggregate check
        https://wiki.yandex-team.ru/solomon/userguide/alerting/#expression

        :param service: service for Juggler aggregate check field
        :param period: sensor aggregation period  :class:`~datetime.timedelta`
        :param program: program in Solomon expression language https://wiki.yandex-team.ru/solomon/userguide/el/
        :param description: will be displayed in Juggler aggregate check, can contain mustache template https://wiki.yandex-team.ru/solomon/userguide/alerting/#templates
        :param juggler_host: host for Juggler aggregate check field
        :param group_by_labels: labels to group Solomon sensors
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """

        check = self.juggler_check_generator.some(service)
        check.children = juggler_children

        alert = self.add_solomon_alert(self.alert_creator.create_expression_alert(
            name="{}.{}".format(check.host, service),
            period=period,
            program=program,
            juggler_description=description,
            juggler_host=solomon_juggler_host,
            juggler_service=check.service,
            group_by_labels=group_by_labels,
        ))

        return check.add_solomon_url(alert)

    def create_expression_alert_check(
        self,
        service,
        period,
        program,
        description,
    ):
        raw_host = "{}-raw".format(self.juggler_check_generator.default_config.host or self.juggler_check_generator.default_config.child_group)
        return self._create_expression_alert_check(
            service,
            period,
            program,
            description,
            solomon_juggler_host=raw_host,
            group_by_labels=[],
            juggler_children=[juggler_sdk.Child(
                host=raw_host,
                service=service,
                group_type=consts.GroupType.host,
            )],
        ).reset_unreachable()

    def create_expression_alert_check_by_host(
        self,
        service,
        period,
        program,
        description,
    ):
        return self._create_expression_alert_check(
            service,
            period,
            program,
            description,
            solomon_juggler_host="{{ labels.host }}",
            group_by_labels=["host"],
            juggler_children=[juggler_sdk.Child(
                host=self.juggler_check_generator.default_config.child_group,
                service=service,
                group_type=self.juggler_check_generator.default_config.child_group_type,
            )],
        )

    def _create_sensor_aggregation_check_program(
        self,
        threshold,
        predicate,
        time_aggregation="avg",
        sensor_glob=None,
        labels=None,
    ):
        """
        Add alert for an aggregated sensor
        :param service:  service for Juggler aggregate check field
        :param period: sensor aggregation period  :class:`~datetime.timedelta`
        :param sensor_glob: string that will used as value for 'sensor' label
        :param labels: dict of labels
        :param threshold: int
        :param predicate: :class:`~crypta.lib.python.solomon.proto.alert_pb2.ECompare`
        :param time_aggregation: https://wiki.yandex-team.ru/solomon/userguide/el/#aggregation-functions
        :param description: will be displayed in Juggler aggregate check, can contain mustache template https://wiki.yandex-team.ru/solomon/userguide/alerting/#templates
        {{ expression.total }} and {{ expression.threshold }} are available in template
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """
        serialized_labels = self.get_serialized_labels(self.get_labels(sensor_glob, labels))

        return """
            let total = {time_aggregation}(
                group_lines("sum",
                    group_by_time(1m, "sum", {labels})
                )
            );
            let threshold = {threshold};

            alarm_if(total {predicate} threshold);
            """.format(time_aggregation=time_aggregation, labels=serialized_labels, threshold=threshold, predicate=solomon_alert_utils.PREDICATE_TO_STRING[predicate])

    def create_sensor_aggregation_check(
        self,
        service,
        period,
        threshold,
        predicate,
        description,
        time_aggregation="avg",
        sensor_glob=None,
        labels=None,
    ):
        return self.create_expression_alert_check(
            service=service,
            period=period,
            program=self._create_sensor_aggregation_check_program(
                sensor_glob=sensor_glob,
                labels=labels,
                threshold=threshold,
                predicate=predicate,
                time_aggregation=time_aggregation,
            ),
            description=description,
        )

    def create_sensor_aggregation_check_by_host(
        self,
        service,
        period,
        threshold,
        predicate,
        description,
        time_aggregation="avg",
        sensor_glob=None,
        labels=None,
    ):
        labels = self.get_labels(sensor_glob, labels)
        labels.setdefault("host", "*")

        return self.create_expression_alert_check_by_host(
            service=service,
            period=period,
            program=self._create_sensor_aggregation_check_program(
                labels=labels,
                threshold=threshold,
                predicate=predicate,
                time_aggregation=time_aggregation,
            ),
            description=description,
        )

    def create_hist_check(
        self,
        service,
        period,
        labels_all,
        labels_bins,
        threshold,
        predicate,
        description,
    ):
        """
        Add alert for a histogram sensor
        :param labels_all: labels for all bins of histogram, string that will used as value for 'sensor' label or dict of labels
        :param labels_bins: labels for specific bins of histogram, string that will used as value for 'sensor' label or dict of labels
        :param threshold: percentile, float
        :param predicate: :class:`~crypta.lib.python.solomon.proto.alert_pb2.ECompare`
        :param description: will be displayed in Juggler aggregate check, can contain mustache template https://wiki.yandex-team.ru/solomon/userguide/alerting/#templates
        {{ expression.percentile }} and {{ expression.threshold }} are available in template
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """
        return self.create_expression_alert_check(
            service,
            period,
            program="""
            let total = group_lines("sum",
                group_by_time(1m, "sum", {labels_all})
            );
            let values = group_lines("sum",
                group_by_time(1m, "sum", {labels_bins})
            );
            let percentile = avg(values / total);
            let threshold = {threshold};

            warn_if(percentile {predicate} threshold);
            """.format(
                labels_all=self.get_serialized_labels(labels_all),
                labels_bins=self.get_serialized_labels(labels_bins),
                threshold=threshold,
                predicate=solomon_alert_utils.PREDICATE_TO_STRING[predicate],
            ),
            description=description,
        )

    @staticmethod
    def get_labels(sensor_glob=None, labels=None):
        if (sensor_glob is None) == (labels is None):
            raise Exception("Only one of (sensor_glob, labels) should be defined, sensor_glob: {}, labels: {}".format(sensor_glob, labels))
        labels = labels or {}
        if sensor_glob is not None:
            labels = {"sensor": sensor_glob}
        return labels

    def get_serialized_labels(self, labels):
        labels = dict(self.alert_creator.selectors, **labels)
        return "{{{}}}".format(",".join('{}="{}"'.format(name, value) for name, value in six.iteritems(labels)))

    # TODO(cherenkov-p-a) maybe move to separate module
    def _create_logbroker_lag_check(self, consumer, topic, threshold, period, dc=None):
        """
        Add alert for read lag
        :param consumer: Logbroker consumer that reads topic
        :param topic: Logbroker topic
        :param threshold: int
        :param period: sensor aggregation period  :class:`~datetime.timedelta`
        :param dc: if None, treat as an lbkx topic, otherwise monitor logbroker topic from this specific dc
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """
        check = self.juggler_check_generator.direct("lag.{}.{}".format(consumer, topic).replace("@", "/"))

        alert = self.add_solomon_alert(solomon_alert_utils.create_threshold_alert(
            name="{}.{}".format(check.host, check.service),
            project_id=self.alert_creator.project_id,
            time_aggregation=alert_pb2.AVG,
            predicate=alert_pb2.GT,
            threshold=threshold,
            period=period,
            selectors={
                "project": "kikimr",
                "cluster": "lbkx" if dc is None else "lbk",
                "service": "pqtabletAggregatedCounters",
                "sensor": "MessageLagByCommitted",
                "ConsumerPath": consumer,
                "TopicPath": topic,
                "user_counters": "PersQueue",
                "OriginDC": "Kafka-bs" if dc is None else dc.capitalize(),
                "host": "dc-unknown" if dc is None else dc.capitalize(),
            },
            juggler_host=check.host,
            juggler_service=check.service,
            juggler_description="lag (msgs): {{ pointValue }}",
        ))

        return check.add_solomon_url(alert)


class ServiceSolomonCheckGenerator(SolomonCheckGenerator):
    """
    SolomonCheckGenerator with solomon shard labels
    """
    def __init__(
        self,
        juggler_check_generator,
        project,
        service,
        cluster,
    ):
        if juggler_check_generator.default_config.host is None:
            juggler_check_generator = juggler_check_generator.clone(host="{}.{}.{}".format(cluster, project, service))

        super(ServiceSolomonCheckGenerator, self).__init__(
            juggler_check_generator,
            solomon_alert_utils.AlertCreator(project, {
                "project": project,
                "service": service,
                "cluster": cluster,
            })
        )

    def create_logbroker_lag_check(
        self,
        consumer,
        topic,
        threshold,
        period,
    ):
        return self._create_logbroker_lag_check(consumer, topic, threshold, period)


class DcSolomonCheckGenerator(SolomonCheckGenerator):
    """
    SolomonCheckGenerator with solomon shard labels and dc label
    """
    def __init__(
        self,
        juggler_check_generator,
        project,
        service,
        cluster,
        dc,
        additional_selectors=None,
    ):
        super(DcSolomonCheckGenerator, self).__init__(
            juggler_check_generator,
            solomon_alert_utils.AlertCreator(project, dict(
                additional_selectors or {},
                project=project,
                service=service,
                cluster=cluster,
                dc=dc,
            ))
        )

    def create_logbroker_lag_check(
        self,
        consumer,
        topic,
        threshold,
        period,
    ):
        return self._create_logbroker_lag_check(consumer, topic, threshold, period, dc=self.alert_creator.selectors["dc"])
