import six

from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.solomon.solomon_alert_registry import SolomonAlertRegistry


class SolomonSensor(SolomonAlertRegistry):
    """
    Representation of a single sensor in solomon
    """
    def __init__(self, juggler_check_generator, alert_creator, sensor):
        super(SolomonSensor, self).__init__()

        self.juggler_check_generator = juggler_check_generator
        self.alert_creator = alert_creator

        if isinstance(sensor, six.string_types):
            self.selector = {"sensor": sensor}
            self.juggler_service = sensor
        elif isinstance(sensor, dict):
            self.selector = sensor
            self.juggler_service = "__".join("{}_{}".format(key, sensor[key]) for key in sorted(sensor.keys()))

    def create_threshold_check(
        self,
        aggregation,
        predicate,
        threshold,
        period,
        description,
        juggler_service=None,
    ):
        """
        Create Solomon threshold alert and Juggler aggregate check
        See create_threshold_alert for argument description
        :return: :class:`~crypta.lib.python.spine.juggler.juggler_aggregate_check.JugglerAggregateCheck`
        """
        juggler_service = juggler_service or self.juggler_service
        alert = self.create_threshold_alert(aggregation, predicate, threshold, period, description, juggler_service)

        return self.juggler_check_generator.some(juggler_service).add_solomon_url(alert)

    def create_threshold_alert(
        self,
        aggregation,
        predicate,
        threshold,
        period,
        description,
        juggler_service=None,
    ):
        """
        Create Solomon threshold alert
        https://wiki.yandex-team.ru/solomon/userguide/alerting/#threshold
        :param aggregation: :class:`~crypta.lib.python.solomon.proto.alert_pb2.EThresholdType`
        :param predicate: :class:`~crypta.lib.python.solomon.proto.alert_pb2.ECompare`
        :param threshold: int
        :param period: sensor aggregation period  :class:`~datetime.timedelta`
        :param description: will be displayed in Juggler aggregate check, can contain mustache template https://wiki.yandex-team.ru/solomon/userguide/alerting/#templates
        :param juggler_service: if defined, will override auto-generated juggler service
        :return: :class:`~crypta.lib.python.solomon.proto.alert_pb2.TAlert`
        """
        config = self.juggler_check_generator.default_config
        is_host = (config.child_group_type == consts.GroupType.host)
        host = config.child_group if is_host else "{{ labels.host }}"

        return self.add_solomon_alert(self.alert_creator.create_threshold_alert(
            selectors=self.selector,
            name="{}.{}".format(config.child_group, juggler_service),
            time_aggregation=aggregation,
            predicate=predicate,
            threshold=threshold,
            period=period,
            juggler_host=host,
            group_by_labels=[] if is_host else ["host"],
            juggler_service=juggler_service,
            juggler_description=description,
        ))
