import datetime

from yt.wrapper import ypath

from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.config_registry import ConfigRegistry
from crypta.lib.python.spine.solomon import (
    solomon_alert_registry,
    solomon_alert_utils,
)
from crypta.lib.python.yt import metrics


class YtMetric(ConfigRegistry):
    """
    Base class for adding yt metrics delivery to solomon, solomon alerts and juggler checks for them
    Metrics are delivered to Solomon
    - project is crypta_stats
    - service is defined in child classes
    - cluster is equal to environment
    """

    _DEFAULT_PERIOD = datetime.timedelta(hours=1)

    def __init__(self, yt_config_registry, path, yt_proxy=YtProxy.hahn, envs=environment.ALL, is_recursive=False):
        """
        :param yt_config_registry: :class:`~crypta.lib.python.spine.yt.yt_config_registry.YtConfigRegistry`
        Adds self as subregistry of yt_config_registry
        :param path: YT path relative to roots in yt_config_registry
        :param yt_proxy: YT cluster
        :param envs: environments for which metrics should be pushed to Solomon
        """
        super(YtMetric, self).__init__()
        self.yt_config_registry = yt_config_registry
        yt_config_registry.add_subregistry(self)
        self.solomon_alert_registry = solomon_alert_registry.SolomonAlertRegistry()
        self.add_subregistry(self.solomon_alert_registry)
        self.yt_proxy = yt_proxy
        self.path = path
        self.used_juggler_services = set()

        for env in envs:
            self.store(self.REGISTRY_TAG, metrics.YtMetricConfig(
                proxy=self.yt_proxy,
                root=self.yt_config_registry.root_paths[env],
                path=self.path,
                env=env,
                project=self.yt_config_registry.solomon_project,
                is_recursive=is_recursive,
            ))

    def _add_alert(self, service, metric, humanize_func, threshold, predicate, env, period=None):
        juggler_service = self._get_juggler_service(env, metric)

        assert juggler_service not in self.used_juggler_services
        self.used_juggler_services.add(juggler_service)

        self.solomon_alert_registry.add_solomon_alert(self._create_solomon_alert(
            service,
            metric,
            humanize_func,
            self.path,
            predicate,
            threshold,
            env,
            period=period,
        ))
        return self._create_juggler_aggregate_check(metric, env)

    def _get_juggler_service(self, env, metric):
        return "{}.{}.{}.{}".format(self.yt_proxy.split(".", 1)[0], env, self.path, metric)

    def _get_juggler_host(self, env):
        return "crypta.{}.{}".format(env, self.yt_proxy)

    def _create_solomon_alert(self, service, metric, humanize_func, path, predicate, threshold, env, period=None):
        project_id = self.yt_config_registry.solomon_project
        selectors = {
            "project": project_id,
            "service": service,
            "metric": metric,
            "sensor": path,
            "cluster": env,
            "yt_proxy": self.yt_proxy,
        }
        description = "{}: {} {} {}".format(
            path,
            metric.replace("_", " "),
            solomon_alert_utils.PREDICATE_TO_STRING[predicate],
            humanize_func(threshold)
        )
        juggler_service = self._get_juggler_service(env, metric)

        return solomon_alert_utils.create_threshold_alert(
            name=juggler_service,
            project_id=project_id,
            time_aggregation=alert_pb2.MAX,
            predicate=predicate,
            threshold=threshold,
            period=period or self._DEFAULT_PERIOD,
            selectors=selectors,
            group_by_labels=[],
            juggler_service=juggler_service,
            juggler_host="crypta.{}.{{{{ labels.yt_proxy }}}}".format(env),
            juggler_description=description,
        )

    def get_full_path(self, env):
        return ypath.ypath_join(self.yt_config_registry.root_paths[env], self.path)

    def _create_juggler_aggregate_check(self, metric, env):
        return self.yt_config_registry.juggler_check_generator.any(
            self._get_juggler_service(env, metric)
        ).add_yt_dependencies(
            [self.yt_proxy]
        ).set_child(
            "crypta.{}.{}".format(env, self.yt_proxy)
        ).add_yt_url(
            self.yt_proxy,
            self.get_full_path(env),
        )
