import datetime

from crypta.lib.python.spine.config_registry import ConfigRegistry
from crypta.lib.python.spine.juggler import (
    consts,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams


class Table(object):
    # TODO(cherenkov-p-a) name duplicates proxy, need to remove something
    def __init__(self, name, proxy, path, expected_attributes=None):
        """
        YT table config
        :param name: name of YT cluster
        :param proxy: YT cluster proxy FQDN
        :param path: table path
        :param expected_attributes: will alert if these attributes differ from expected values
        """
        self.name = name
        self.proxy = proxy
        self.path = path
        self.expected_attributes = expected_attributes or {}

    def get_raw_juggler_host(self):
        return "crypta-yt-replicated-table-checker"

    def get_juggler_service(self):
        raise NotImplementedError

    def dump(self):
        return self.__dict__


class Master(Table):
    """
    Replicated table config
    https://docs.yandex-team.ru/yt/description/dynamic_tables/replicated_dynamic_tables
    Daemon will check that
    - it is a dynamic "replicated_table"
    - it is mounted and has all correct expected_attributes
    - it has no replication errors
    - its list of replicas is the same as in config
    - all replicas are enabled
    - replication lag time is below threshold
    - number of synchronous replicas is correct
    """
    def __init__(self, name, proxy, path, replication_lag_threshold, sync_count, expected_attributes=None):
        """
        :param name: name of YT cluster
        :param proxy: YT cluster proxy FQDN
        :param path: table path
        :param replication_lag_threshold: :class:`~datetime.timedelta`
        :param sync_count: expected number of synchronous replicas
        :param expected_attributes: will alert if these attributes differ from expected values
        """
        super(Master, self).__init__(name, proxy, path, expected_attributes)
        self.replication_lag_threshold = replication_lag_threshold
        self.sync_count = sync_count

    def get_juggler_service(self):
        return "master.{}.{}".format(self.name, self.path)


class Replica(Table):
    """
    Replica config
    https://docs.yandex-team.ru/yt/description/dynamic_tables/sorted_dynamic_tables
    Daemon will check that
    - it is a dynamic table
    - it is mounted and has all correct expected_attributes
    """
    def __init__(self, name, proxy, path, expected_attributes=None, expected_replication_attributes=None):
        super(Replica, self).__init__(name, proxy, path, expected_attributes)
        self.expected_replication_attributes = expected_replication_attributes or {}

    def get_juggler_service(self):
        return "replica.{}.{}".format(self.name, self.path)


class ReplicatedTable(object):
    REGISTRY_TAG = "yt_replicated_table"

    def __init__(self, master, replicas):
        """
        Replicated table with corresponding replicas
        :param master: :class:`Master`
        :param replicas: list of :class:`Replica`
        """
        super(ReplicatedTable, self).__init__()
        self.master = master
        self.replicas = replicas

    def dump(self):
        return {"master": self.master.dump(), "replicas": [replica.dump() for replica in self.replicas]}


class ReplicatedTableConfigRegistry(ConfigRegistry):
    def __init__(self, juggler_check_generator, replicated_table):
        """
        Config registry that keeps ReplicatedTable config and its corresponding juggler checks
        :param juggler_check_generator: :class:`~crypta.lib.python.spine.juggler.juggler_check_generator.JugglerCheckGenerator`
        :param replicated_table: :class:`ReplicatedTable`
        """
        super(ReplicatedTableConfigRegistry, self).__init__()

        juggler_check_generator = juggler_check_generator.clone(
            child_group=replicated_table.master.get_raw_juggler_host(),
            child_group_type=consts.GroupType.host,
        )

        flap_detector = FlapDetectorParams(
            datetime.timedelta(minutes=10),
            datetime.timedelta(minutes=30),
        )

        for replica in replicated_table.replicas:
            juggler_check_generator.any(replica.get_juggler_service()).add_flap_detector(flap_detector).add_yt_url(
                replica.proxy,
                replica.path,
            )

        juggler_check_generator.any(replicated_table.master.get_juggler_service()).add_flap_detector(
            flap_detector
        ).add_yt_url(
            replicated_table.master.proxy,
            replicated_table.master.path,
        )

        self.store(ReplicatedTable.REGISTRY_TAG, replicated_table)

        juggler_check_generator.add_subregistry(self)
