# coding: utf-8

from collections import defaultdict
import logging
import os
import six

from statface_client import (
    StatfaceClient,
    StatfaceReportConfig,
)
from statface_client.constants import (
    STATFACE_PRODUCTION_UPLOAD,
    STATFACE_BETA,
)

FIELDDATE = six.u("fielddate")

logger = logging.getLogger(__name__)


def upload_to_statface(report_path, data, scale, is_production=False):
    """Upload data to report. Uses oauth token from STATFACE_TOKEN environment variable; otherwise uses contents
      of ~/.statbox/statface_auth.yaml.

    required:
    :param report_path: (string) report path, e.g. "Adhoc/username/report_name"
    :param data: (list of dicts) data to upload, measures AND dimensions in Statface terms; remember to add "fielddate" to every dict.
    :param scale: (string) one of STATFACE_SCALES; for corresponding required formats of fielddate,
      see https://wiki.yandex-team.ru/statbox/Statface/externalreports/#oboznachenijamasshtabovscaleotchjota.

    optional:
    :param is_production: (bool) if True, send data to the production installation of Statface instead of test one.
    """

    report = get_statface_report(report_path, is_production)

    logger.debug("Data: %s", data)
    report.upload_data(scale=scale, data=data)


def create_or_update_report(report_path, report_title, measures, additional_dimensions, titles_dict, is_production=False):
    """Creates or updates report at report_path according to keys in data

    :param report_path: report path
    :param report_title: report_title
    :param measures: (list of strings) measures (numeric values in report)
    :param additional_dimensions: (list of strings) additional dimensions in report ("fielddate" is added automatically)
    :param measures: (list of dicts) data to upload, measures AND dimensions in Statface terms; remember to add "fielddate" to every dict.
    :param titles_dict: a mapping from data keys to human-readable column names; must contain all data keys or be empty.
    :param is_production: use production installation of Statface if True
    """

    report = get_statface_report(report_path, is_production)

    report_config_as_yaml = get_report_config_as_yaml(measures, additional_dimensions, titles_dict)
    logger.info("Report config: %s", report_config_as_yaml)

    config = StatfaceReportConfig()
    config.from_yaml(report_config_as_yaml)
    config.title = six.text_type(report_title)
    report.upload_config(config)


def get_statface_report(report_path, is_production):
    logger.info("Report path: %s", report_path)

    client_config = get_client_config(is_production)
    logger.info("Statface token: %s",  get_masked_token(client_config["oauth_token"]))
    logger.info("Statface host: %s", client_config["host"])

    client = StatfaceClient(client_config=client_config)
    return client.get_report(report_path)


def get_report_config_as_yaml(measures, additional_dimensions, titles_dict, additional_dimension_types=defaultdict(lambda: "number")):
    CONFIG_TEMPLATE = six.u("""---
dimensions:
{dimensions_config}
measures:
{measure_types_config}
titles:
{titles_config}
""")
    dimensions = [FIELDDATE] + additional_dimensions
    dimension_types = additional_dimension_types.copy()
    dimension_types["fielddate"] = "date"
    dimensions_config = six.u("\n").join([six.u("- {}: {}").format(name, dimension_types[name]) for name in dimensions])

    measure_types_config = six.u("\n").join([six.u("- {}: number").format(name) for name in measures])

    data_names = dimensions + measures
    titles_config = get_titles_config(data_names, titles_dict)

    return CONFIG_TEMPLATE.format(
        dimensions_config=dimensions_config,
        measure_types_config=measure_types_config,
        titles_config=titles_config,
    )


def get_titles_config(data_names, titles_dict):
    if titles_dict:
        title_keys = set(titles_dict.keys())
        names = set(data_names)
        if names != title_keys:
            raise Exception("Metric names and title keys differ.\n  Names: {}\n  Titles keys: {}".format(sorted(names), sorted(title_keys)))

        titles = [(name, titles_dict[name]) for name in data_names]
    else:
        titles = [(name, name) for name in data_names]

    return six.text_type(six.u("\n").join([six.u("  {}: {}").format(name, title) for name, title in titles]))


def get_client_config(is_production):
    return {
        "host": get_statface_host(is_production),
        "oauth_token": os.environ.get("STATFACE_TOKEN", "")
    }


def get_statface_host(is_production):
    return STATFACE_PRODUCTION_UPLOAD if is_production else STATFACE_BETA


def get_masked_token(token, min_len=9, visible_chars=6):
    visible_chars = min(visible_chars, len(token))
    if len(token) >= min_len:
        return token[:visible_chars] + ("X" * (len(token) - visible_chars))
    return "X" * len(token)
