import bravado_core.response  # noqa

from google.protobuf.message_factory import MessageFactory

assert bravado_core.version == '5.17.0'
old_unmarshal_response = bravado_core.response.unmarshal_response


def my_unmarshal_response(response, op):
    """Unmarshal incoming http response into a value based on the
    response specification.
    :type response: :class:`bravado_core.response.IncomingResponse`
    :type op: :class:`bravado_core.operation.Operation`
    :returns: value where type(value) matches response_spec['schema']['type']
        if it exists, None otherwise.
    """
    from bravado_core.response import (
        get_response_spec,
        validate_schema_object,
        unmarshal_schema_object,
        msgpack,
        APP_JSON,
        APP_MSGPACK,
    )
    deref = op.swagger_spec.deref
    response_spec = get_response_spec(response.status_code, op)

    if 'schema' not in response_spec:
        # If response spec does not define schema
        return None

    content_type = response.headers.get('content-type', '').lower()

    if content_type.startswith(APP_JSON) or content_type.startswith(APP_MSGPACK):
        content_spec = deref(response_spec['schema'])
        # MONKEY PATCH WARNING
        # We handle file as raw text
        if content_spec['type'] == 'file':
            return response._delegate.content
        # END
        if content_type.startswith(APP_JSON):
            try:
                content_value = response.json()
            except:
                logger.error('Failed to parse response: {%s}', response.text)
                content_value = None
        else:
            content_value = msgpack.loads(response.raw_bytes, raw=False)
        if op.swagger_spec.config['validate_responses']:
            validate_schema_object(op.swagger_spec, content_spec, content_value)

        return unmarshal_schema_object(op.swagger_spec, content_spec, content_value)

    # TODO: Non-json response contents
    return response.text
bravado_core.response.unmarshal_response = my_unmarshal_response


import logging  # noqa
logger = logging.getLogger(__name__)

from bravado.requests_client import (
    RequestsClient, RequestsFutureAdapter
)  # noqa
from bravado.client import SwaggerClient  # noqa
from bravado.config import CONFIG_DEFAULTS  # noqa
from bravado.exception import (
        HTTPTooManyRequests,  # noqa
        HTTPServiceUnavailable,  # noqa
        HTTPGatewayTimeout,  # noqa
        HTTPBadGateway,  # noqa
)  # noqa
from retry import retry  # noqa

CONFIG_DEFAULTS['also_return_response'] = True

RETRIABLE_ERRORS = (
    HTTPTooManyRequests,
    HTTPServiceUnavailable,
    HTTPBadGateway,
    HTTPGatewayTimeout
)

RETRY_CONFIG = {
    'tries': 10,
    'delay': 1,
    'backoff': 1.259,
    'logger': logger
}


class OAuthAuthenticator(object):

    def __init__(self, token):
        self.token = token

    def apply(self, request):
        request.headers['Authorization'] = 'OAuth {}'.format(self.token)
        return request

    def matches(self, url):
        return True


class TracingRequestsClient(RequestsClient):

    def __init__(self, token, *args, **kwargs):
        super(TracingRequestsClient, self).__init__(*args, **kwargs)
        self.authenticator = OAuthAuthenticator(token)

    def request(self, *args, **kwargs):
        return TracingFuture(
            super(TracingRequestsClient, self).request(*args, **kwargs)
        )


class TracingFuture(RequestsFutureAdapter):

    def __init__(self, delegate):
        self.delegate = delegate

    @retry(RETRIABLE_ERRORS, **RETRY_CONFIG)
    def retrieve_result_with_retries(self, *args, **kwargs):
        return self.delegate.result(*args, **kwargs)

    def result(self, *args, **kwargs):
        result = self.retrieve_result_with_retries(*args, **kwargs)

        if isinstance(result, tuple):
            value, response = result
            logger.info('Request [%s] for %s got %d',
                        response.headers.get('X-Crypta-Request-ID', 'NO-ID'),
                        response._delegate.request.url,
                        response.status_code)
            return value
        else:
            return result


def swagger(swagger_json_url, token):
    import requests.packages.urllib3 as urllib3
    urllib3.disable_warnings()
    http_client = TracingRequestsClient(token)
    http_client.session.verify = False
    client = SwaggerClient.from_url(swagger_json_url, http_client=http_client)
    return client


def _to_camelcase(name):
    return ''.join(word.title() if i else word for i, word in enumerate(name.split('_')))


def _to_proto(clazz, swagger_object):
    factory = MessageFactory()
    kwargs = {}
    for field in clazz.DESCRIPTOR.fields:
        if field.message_type:
            subclazz = factory.GetPrototype(field.message_type)
            submessage = getattr(swagger_object, _to_camelcase(field.name), None)
            if submessage:
                kwargs[field.name] = _to_proto(subclazz, submessage)
        else:
            value = getattr(swagger_object, _to_camelcase(field.name), None)
            if value is not None:
                kwargs[field.name] = value
    return clazz(**kwargs)
