import requests


class YasmClient(object):
    def __init__(self, url_prefix="https://yasm.yandex-team.ru/srvambry"):
        self.url_prefix = url_prefix

    def _url(self, path):
        return self.url_prefix + path

    # https://wiki.yandex-team.ru/golovan/userdocs/templatium/panels/api/
    def _tmpl(self, command):
        return self._url("/tmpl/panels/" + command)

    def create_template(self, key, owners, abc, content):
        return _post(self._tmpl("create"), json={
            "abc": abc,
            "key": key,
            "owners": owners,
            "content": content,
        })

    def update_template(self, key, owners, abc, content):
        return _post(self._tmpl("update"), params={"key": key}, json={
            "abc": abc,
            "owners": owners,
            "content": content,
        })

    def list_templates(self, abc):
        return _get(self._tmpl("list"), params={"abc": abc})

    def delete_template(self, key):
        return _post(self._tmpl("delete"), json={"key": key})

    def get_template(self, key):
        resp = requests.get(self._tmpl("get"), params={"key": key})
        if resp.status_code == requests.codes.not_found:
            return None

        resp.raise_for_status()
        return resp.json()["response"]

    # https://wiki.yandex-team.ru/golovan/userdocs/panels/menutree/api
    def _menutree(self, command):
        return self._url("/menutree/" + command)

    def replace_menutree(self, path, tree):
        return _post(self._menutree("replace"), json={
            "path": path,
            "tree": tree,
        })


def _check_for_errors(response):
    if not response.ok:
        raise Exception(response.json()["error"])


def _post(*args, **kwargs):
    response = requests.post(*args, **kwargs)
    _check_for_errors(response)
    return response.json()


def _get(*args, **kwargs):
    response = requests.get(*args, **kwargs)
    _check_for_errors(response)
    return response.json()["response"]
