import logging
import os

import ydb


logger = logging.getLogger(__name__)


class YdbClient(object):
    def __init__(self, endpoint, database, token=None):
        self.endpoint = endpoint
        self.database = database

        self.driver = ydb.Driver(
            ydb.ConnectionParams(self.endpoint, self.database, auth_token=token),
        )
        self.driver.wait()

    @classmethod
    def from_proto(cls, config):
        return cls(endpoint=config.Endpoint, database=config.Database, token=config.Token)

    def create_session(self):
        session = self.driver.table_client.session()
        session.create()
        return session

    def get_full_path(self, path):
        if path.startswith(self.database):
            return path

        return os.path.join(self.database, path)

    def list_directory(self, path=None):
        path = self.get_full_path(path) if path is not None else self.database
        scheme_entry = self.driver.scheme_client.list_directory(path)
        assert scheme_entry.is_directory_or_database(), "{} is not directory or database".format(path)
        return scheme_entry.children

    def make_directory(self, path):
        path = self.get_full_path(path)
        self.driver.scheme_client.make_directory(path)

    def remove_directory(self, path, recursive=False):
        if self.is_system_directory(path):
            return

        items = self.list_directory(path)

        assert recursive or not items, "Can't remove directory with content without recursive=True"

        for item in items:
            item_path = os.path.join(path, item.name)
            if item.is_table():
                self.drop_table(item_path)
            if item.is_directory():
                self.remove_directory(item_path, recursive)

        self.driver.scheme_client.remove_directory(self.get_full_path(path))

    def describe_table(self, path):
        path = self.get_full_path(path)
        scheme_entry = self.create_session().describe_table(path)
        assert scheme_entry.is_table()

        return scheme_entry

    def create_table(self, path, columns, primary_key, profile=None):
        path = self.get_full_path(path)
        primary_key = primary_key if isinstance(primary_key, list) else [primary_key]
        table_description = ydb.TableDescription().with_columns(*columns).with_primary_keys(*primary_key)

        if profile is not None:
            table_description.with_profile(profile)

        self.create_session().create_table(path, table_description)

    def copy_table(self, source_path, destination_path):
        assert not self.is_path_exists(destination_path)

        source_path = self.get_full_path(source_path)
        destination_path = self.get_full_path(destination_path)
        self.create_session().copy_table(source_path, destination_path)

    def drop_table(self, path):
        path = self.get_full_path(path)
        self.create_session().drop_table(path)

    def is_path_exists(self, path):
        path = self.get_full_path(path)
        try:
            self.driver.scheme_client.describe_path(path)
            return True
        except ydb.SchemeError:
            return False

    def is_system_directory(self, path):
        return self.get_full_path(path) == os.path.join(self.database, ".sys")
