# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import unicode_literals
import collections
import contextlib
import json
import logging
import re
import os

from cached_property import cached_property
from library.python import resource
import six
from textwrap import dedent
from yql.api.v1.client import YqlClient
from yt import yson
import yt.wrapper as yt

from crypta.lib.python import (
    templater,
    script_name,
)
from crypta.lib.python.yt.yt_helpers import tempdir


logger = logging.getLogger(__name__)


TEMPLATE = u'''-- <generated>
{% for pragma in pragmas %}
{{pragma}}
{% endfor %}

PRAGMA yt.MaxJobCount = "{{max_job_count}}";{{max_job_count_description}}

{% for lib in yql_libs %}
{{lib}}
{% endfor %}
-- </generated>

{{query}}'''

TEST_ENV = [
    ("CRYPTA_TEST_YQL_SERVER", "yql_server", str),
    ("CRYPTA_TEST_YQL_PORT", "yql_port", int),
    ("CRYPTA_TEST_YQL_DB", "db", str)
]

Udf = collections.namedtuple("Udf", ["so_name", "url"])


def get_test_environment():
    test_vars = {}
    for env_var, config_key, convertor in TEST_ENV:
        if env_var in os.environ:
            logger.info("Detected test env var '%s' which replaces '%s'. Value: %s", env_var, config_key, os.environ[env_var])
            test_vars[config_key] = convertor(os.environ[env_var])

    if len(test_vars) == len(TEST_ENV):
        logger.info("All test env vars were defined, YQL executer will use them.")
        return test_vars
    elif len(test_vars):
        logger.info("Some but not all test env vars were defined, not using them.")
        return {}
    else:
        logger.info("Running in normal environment")
        return {}


def sandbox_mark(title):
    if os.getenv('YQL_MARK'):
        # title should has YQL substring, so if title is None or empty, set 'Crypta YQL'
        return '{title} {mark}'.format(title=(title or 'Crypta YQL'), mark=os.getenv('YQL_MARK'))
    return title


def pragma_transaction(transaction):
    if transaction:
        return 'PRAGMA yt.ExternalTx="{transaction}";'.format(
            transaction=transaction)
    else:
        return '-- no transaction'


def pragma_pool(pool):
    if pool:
        return 'PRAGMA yt.StaticPool="{pool}";'.format(pool=pool)
    else:
        return '-- default pool'


def pragma_weight(weight):
    if weight:
        return 'PRAGMA yt.DefaultOperationWeight="{weight}";'.format(weight=weight)
    else:
        return '-- default weight'


def pragma_operation_spec(operation_spec):
    if operation_spec:
        spec = six.ensure_str(yson.dumps(operation_spec))
        spec = spec.replace("'", "\\'")
        return "PRAGMA yt.OperationSpec='{operation_spec}';".format(operation_spec=spec)
    else:
        return '-- default operation spec'


def pragma_tmp_folder(tmp_folder):
    if tmp_folder:
        return 'PRAGMA yt.TmpFolder="{tmp_folder}";'.format(
            tmp_folder=tempdir.to_absolute_path(tmp_folder))
    else:
        return '-- default tmp folder'


def pragma_erasure_codec(erasure_codec):
    return 'PRAGMA yt.PublishedErasureCodec="{}";'.format(erasure_codec) if erasure_codec else '-- default erasure codec'


def pragma_compression_codec(compression_codec):
    return 'PRAGMA yt.PublishedCompressionCodec="{}";'.format(compression_codec) if compression_codec else '-- default compression codec'


def pragma_udf(udf):
    return dedent("""
        PRAGMA File('{so_name}', '{url}');
        PRAGMA udf('{so_name}');
    """).format(**udf._asdict()).strip()


def pragma_disable_query_cache(disable_query_cache):
    if disable_query_cache:
        return 'PRAGMA yt.QueryCacheMode="disable";'
    else:
        return '-- default query cache mode'


def pragma_binary_cache(binary_cache_tmp_folder, binary_cache_ttl):
    if binary_cache_tmp_folder is not None:
        if binary_cache_ttl is not None:
            pragma_ttl = 'PRAGMA yt.BinaryExpirationInterval = "{}s";'.format(int(binary_cache_ttl.total_seconds()))
        else:
            pragma_ttl = '-- default binary cache ttl'
        return 'PRAGMA yt.BinaryTmpFolder="{}";\n{}'.format(binary_cache_tmp_folder, pragma_ttl)
    else:
        return '-- default binary cache tmp folder'


def yt_name_to_yql_cluster(yt_name):
    return {
        "seneca-sas": "senecasas",
        "seneca-man": "senecaman",
        "seneca-vla": "senecavla"
    }.get(yt_name, yt_name)


def yt_shortname(yt_proxy):
    return yt_proxy.split('.')[0]


def query_with_pragmas(
    query,
    pool,
    transaction,
    tmp_folder,
    weight,
    operation_spec,
    max_job_count=None,
    erasure_codec=None,
    compression_codec=None,
    udfs=None,
    yql_libs=None,
    disable_cache=False,
    disable_cache_with_tmp=True,
    binary_cache_tmp_folder=None,
    binary_cache_ttl=None
):
    max_job_count, max_job_count_description = (max_job_count, "") if max_job_count else (16384, "  -- YQL-6730")
    udfs = udfs or []
    yql_libs = yql_libs or []

    return templater.render_template(TEMPLATE, {
        "pragmas": [
            pragma_transaction(transaction=transaction),
            pragma_pool(pool=pool),
            pragma_tmp_folder(tmp_folder),
            pragma_weight(weight),
            pragma_operation_spec(operation_spec),
            pragma_erasure_codec(erasure_codec),
            pragma_compression_codec(compression_codec),
            pragma_disable_query_cache((tmp_folder is not None and disable_cache_with_tmp) or disable_cache),
            pragma_binary_cache(binary_cache_tmp_folder, binary_cache_ttl),
        ] + [pragma_udf(udf) for udf in udfs],
        "max_job_count": max_job_count,
        "max_job_count_description": max_job_count_description,
        "query": query,
        "yql_libs": yql_libs,
    })


def create_delegate_yql_client(**kwargs):
    from yql.config import config as yql_config
    yql_config.progress_on_stderr = False
    yql_config.legacy_prepare_cell = True
    return YqlClient(**kwargs)


class YqlExecutionException(Exception):
    pass


def get_yt_client(yt_proxy, token, tx_id):
    client = yt.YtClient(yt_proxy, token, config=yt.config.config)
    if tx_id is not None:
        yt.config.set_command_param("transaction_id", tx_id, client)
    return client


@contextlib.contextmanager
def maybe_create_tmp_subfolder(yt_proxy, token, transaction, tmp_folder, delete_tmp=True):
    if tmp_folder and delete_tmp:
        yt_client = get_yt_client(yt_proxy, token, transaction)

        with tempdir.YtTempDir(yt_client, tmp_folder) as tmp_subfolder:
            yield tmp_subfolder.transaction.transaction_id, tmp_subfolder.path
    else:
        yield transaction, tmp_folder


def detect_script_name():
    return script_name.detect_script_name(skip_locations={"crypta/lib/python/yql"})


class CustomYqlClient(object):

    # TODO: add file attachment support

    def __init__(self, yt_proxy, token, transaction=None, db=None, yql_server=None, yql_port=None, pool=None,
                 title=None, weight=None, additional_attributes=None, script_name=None, syntax_version=None,
                 operation_spec=None, tmp_folder=None, max_job_count=None, attached_files=None,
                 disable_cache=False, disable_cache_with_tmp=True,
                 binary_cache_tmp_folder=None, binary_cache_ttl=None):
        """
        :param yt_proxy: str
        :param token: str
        :param transaction: transaction id
        :param db: str
        :param yql_server: str
        :param yql_port: int
        :param pool: str
        :param title: str
        :param weight: double
        :param operation_spec: yson str
        :param additional_attributes: {"key": "value"} will be mapped to yt spec {"description": {"yql_key": "value"}}
        :param script_name: str
        :param syntax_version: int
        :param tmp_folder: str
        :param max_job_count: int
        :param attached_files: [{"name": ..., "content": ..., "disposition": ...}, ...]
        :param disable_cache: bool
        :param disable_cache_with_tmp: bool
        :param binary_cache_tmp_folder: str
        :param binary_cache_ttl: timedelta
        """
        self.yt_proxy = yt_proxy
        self.token = token
        self.transaction = transaction
        self.db = db or yt_shortname(yt_proxy)
        self.yql_server = yql_server
        self.yql_port = yql_port
        self.pool = pool
        self.title = title
        self.weight = weight
        self.additional_attributes = additional_attributes or {}
        self.syntax_version = syntax_version
        self.operation_spec = operation_spec
        self.tmp_folder = tmp_folder
        self.max_job_count = max_job_count
        self.attached_files = attached_files
        self.disable_cache = disable_cache
        self.disable_cache_with_tmp = disable_cache_with_tmp
        self.binary_cache_tmp_folder = binary_cache_tmp_folder
        self.binary_cache_ttl = binary_cache_ttl

        if not script_name:
            script_name = title

        if script_name:
            self.additional_attributes['script_name'] = script_name

    def path(self, yt_path):
        """
        Drops leading // in YT path to make it YQL compatible.
        """
        return re.sub('^//', '', str(yt_path))

    @cached_property
    def client(self):
        delegate_args = dict()
        delegate_args['token'] = self.token
        if self.yql_server:
            delegate_args['server'] = self.yql_server
        if self.yql_port:
            delegate_args['port'] = self.yql_port
        if self.db:
            delegate_args['db'] = yt_name_to_yql_cluster(self.db)
        return create_delegate_yql_client(**delegate_args)

    def query(self, query, pool=None, transaction=None, tmp_folder=None, title=None, weight=None,
                additional_attributes=None, script_name=None, syntax_version=None, operation_spec=None,
                attached_files=None, erasure_codec=None, compression_codec=None, udfs=None, yql_libs=None,
                binary_cache_tmp_folder=None, binary_cache_ttl=None):
        assert query, 'Query should be provided'

        if pool is None:
            pool = self.pool

        if transaction is None:
            transaction = self.transaction

        if tmp_folder is None:
            tmp_folder = self.tmp_folder

        if title is None:
            title = self.title

        if weight is None:
            weight = self.weight

        if additional_attributes is None:
            additional_attributes = dict()

        if self.additional_attributes:
            result_additional_attributes = {}
            result_additional_attributes.update(self.additional_attributes)
            result_additional_attributes.update(additional_attributes)
            additional_attributes = result_additional_attributes

        additional_attributes.setdefault('script_name', script_name or title or 'YQL_{}'.format(detect_script_name()))

        if syntax_version is None:
            syntax_version = self.syntax_version

        if operation_spec is None:
            operation_spec = self.operation_spec

        if attached_files is None:
            attached_files = self.attached_files

        processed_query = query_with_pragmas(
            query,
            pool=pool,
            transaction=transaction,
            tmp_folder=tmp_folder,
            weight=weight,
            operation_spec=operation_spec,
            max_job_count=self.max_job_count,
            erasure_codec=erasure_codec,
            compression_codec=compression_codec,
            udfs=udfs,
            yql_libs=yql_libs,
            disable_cache=self.disable_cache,
            disable_cache_with_tmp=self.disable_cache_with_tmp,
            binary_cache_tmp_folder=binary_cache_tmp_folder or self.binary_cache_tmp_folder,
            binary_cache_ttl=binary_cache_ttl or self.binary_cache_ttl,
        )

        query_kwargs = dict(title=sandbox_mark(title))
        if syntax_version is not None:
            query_kwargs.update(dict(syntax_version=syntax_version))

        handle = self.client.query(processed_query, **query_kwargs)
        if additional_attributes:
            handle.additional_attributes = additional_attributes

        if attached_files:
            for item in attached_files:
                file_content = item["content"]
                if item["disposition"] == "resource":
                    file_content = resource.find(file_content)
                handle.attached_files.append({"name": item["name"], "data": file_content, "type": "CONTENT"})

        return handle

    def execute(self, query, pool=None, transaction=None, tmp_folder=None, title=None, weight=None, parameters=None,
                additional_attributes=None, script_name=None, syntax_version=None, operation_spec=None,
                attached_files=None, erasure_codec=None, compression_codec=None, udfs=None, yql_libs=None,
                binary_cache_tmp_folder=None, binary_cache_ttl=None, delete_tmp=True):
        assert query, 'Query should be provided'
        if transaction is None:
            transaction = self.transaction

        if tmp_folder is None:
            tmp_folder = self.tmp_folder

        with maybe_create_tmp_subfolder(self.yt_proxy, self.token, transaction, tmp_folder, delete_tmp) as (transaction, tmp_folder):
            handle = self.query(query, pool=pool, transaction=transaction, tmp_folder=tmp_folder, title=title, weight=weight,
                                additional_attributes=additional_attributes, script_name=script_name, syntax_version=syntax_version,
                                operation_spec=operation_spec,
                                attached_files=attached_files, erasure_codec=erasure_codec, compression_codec=compression_codec,
                                udfs=udfs, yql_libs=yql_libs,
                                binary_cache_tmp_folder=binary_cache_tmp_folder, binary_cache_ttl=binary_cache_ttl)

            handle.run(parameters=parameters)
            logger.debug('Executing YQL query: [%s]', handle.query)
            logger.info('Executing YQL query with id [%s] by %s and additional attributies: %s',
                        handle.operation_id, handle.json.get('username'), str(additional_attributes))
            logger.info('YQL operation shared URL: %s', handle.share_url)

            results = handle.get_results()

            if handle.is_ok:
                logger.info(json.dumps(handle.json, indent=4))
                logger.info(results)

            if not handle.is_success:
                raise YqlExecutionException(
                    u"\n> Status: {}\n> Errors:\n{}\n> Query:\n{}".format(
                        handle.status,
                        u"\n".join([str(error) for error in handle.errors]),
                        u"\n".join([
                            u"{}\t{}".format(i, l)
                            for i, l in enumerate(handle.query.split("\n"), 1)
                        ])
                    )
                )
            else:
                return list(results)

    def explain(self, query, pool=None, transaction=None, tmp_folder=None, weight=None, additional_attributes=None, syntax_version=None, operation_spec=None):
        assert query, 'Query should be provided'

        if pool is None:
            pool = self.pool

        if transaction is None:
            transaction = self.transaction

        if weight is None:
            weight = self.weight

        if additional_attributes is None:
            additional_attributes = self.additional_attributes

        if syntax_version is None:
            syntax_version = self.syntax_version

        if operation_spec is None:
            operation_spec = self.operation_spec

        with maybe_create_tmp_subfolder(self.yt_proxy, self.token, transaction or self.transaction, tmp_folder) as (transaction, tmp_folder):
            processed_query = query_with_pragmas(
                query,
                pool=pool,
                transaction=transaction,
                tmp_folder=tmp_folder,
                weight=weight,
                operation_spec=operation_spec,
                max_job_count=self.max_job_count,
            )
            logger.debug('Running explain for query: [%s]', processed_query)
            query_kwargs = dict()
            if syntax_version is not None:
                query_kwargs.update(dict(syntax_version=syntax_version))
            handle = self.client.query(processed_query, **query_kwargs).explain()
            if additional_attributes:
                handle.additional_attributes = additional_attributes
            plan, ast = handle.plan, handle.ast
            if not handle.is_success:
                raise YqlExecutionException((handle.status,
                                             [str(error)
                                              for error in handle.errors]))
            else:
                return plan, ast


class CustomYqlEmbeddedClient(object):

    def __init__(self, yt_proxy, token, mrjob_binary, udf_resolver_binary, udfs_dir,
                 title, transaction=None, db=None, yql_server=None, yql_port=None, pool=None, weight=None, additional_attributes=None, script_name=None, syntax_version=None, operation_spec=None,
                 max_job_count=None):
        self.yt_proxy = yt_proxy
        self.token = token
        self.transaction = transaction
        self.db = db or yt_shortname(yt_proxy)
        self.yql_server = yql_server
        self.yql_port = yql_port
        self.pool = pool
        self.mrjob_binary = mrjob_binary
        self.udf_resolver_binary = udf_resolver_binary
        self.udfs_dir = udfs_dir
        self.title = title
        self.weight = weight
        self.additional_attributes = additional_attributes or {}
        if script_name:
            self.additional_attributes['script_name'] = script_name
        self.syntax_version = syntax_version
        self.operation_spec = operation_spec
        self.max_job_count = max_job_count

    @cached_property
    def client(self):
        import yql.library.embedded.python.run as embedded

        yt_clusters = [{
            'name': 'ytcluster',
            'cluster': self.yt_proxy
        }]
        return embedded.OperationFactory(
            yt_clusters=yt_clusters,
            yt_token=self.token,
            mrjob_binary=self.mrjob_binary,
            udf_resolver_binary=self.udf_resolver_binary,
            udfs_dir=self.udfs_dir,
        )

    def execute(self, query, pool=None, transaction=None, tmp_folder=None, weight=None, additional_attributes=None, title=None, script_name=None, syntax_version=None, operation_spec=None):
        assert query, 'Query should be provided'

        if pool is None:
            pool = self.pool

        if transaction is None:
            transaction = self.transaction

        if weight is None:
            weight = self.weight

        if additional_attributes is None:
            additional_attributes = dict()

        if self.additional_attributes:
            additional_attributes.update(self.additional_attributes)

        if syntax_version is None:
            syntax_version = self.syntax_version

        if operation_spec is None:
            operation_spec = self.operation_spec

        if not script_name and not additional_attributes.get('script_name'):
            script_name = title

        if script_name:
            additional_attributes['script_name'] = script_name

        with maybe_create_tmp_subfolder(self.yt_proxy, self.token, transaction, tmp_folder) as (transaction, tmp_folder):
            processed_query = query_with_pragmas(
                query,
                pool=pool,
                transaction=transaction,
                tmp_folder=tmp_folder,
                weight=weight,
                operation_spec=operation_spec,
                max_job_count=self.max_job_count,
            )
            logger.info('Executing YQL query: [%s]', processed_query)

            kwargs = {}
            if syntax_version is not None:
                kwargs.update(syntax_version=syntax_version)

            handle = self.client.run(
                processed_query,
                title=sandbox_mark(self.title),
                attributes=yson.dumps(additional_attributes),
                **kwargs
            )
            return handle.yson_result()


def create_yql_client(yt_proxy, token, **kwargs):
    """ Creates transactional YQL client with pool and YT proxy substitution.

    :param yt_proxy: YT proxy FQDN
    :param token: YQL (!) token
    """
    kwargs.update(get_test_environment())
    return CustomYqlClient(yt_proxy=yt_proxy, token=token, **kwargs)


def create_yql_embedded_client(yt_proxy, token, title, **kwargs):
    """ Creates transactional YQL client with pool and YT proxy substitution.

    :param yt_proxy: YT proxy FQDN
    :param token: YQL (!) token
    """
    return CustomYqlEmbeddedClient(yt_proxy=yt_proxy, token=token, title=title, **kwargs)
