from __future__ import print_function

import time
import sys
import crypta.lib.python.bt.conf.conf as conf

from cached_property import cached_property
from crypta.lib.python.bt.tasks import TransactionalYtTask
from crypta.lib.python.jinja_resource import JinjaResourceMixin
from crypta.lib.python.yql_runner.base_parser import EmbeddedYqlMixin, ClientYqlMixin, sandbox_mark


class ConfMixin(object):

    """Usefull property for get config params"""

    def get_token(self):
        return conf.proto.Yt.Token

    def get_yql_token(self):
        """required for client mode"""
        return conf.proto.Yql.Token or self.get_token()

    @cached_property
    def yt_proxy(self):
        return self.yt_config["proxy"]["url"]

    @cached_property
    def crypta_env(self):
        from crypta.lib.python import getoptpb

        return getoptpb.convert_env_enum_to_string(conf.proto.Environment)

    @cached_property
    def udf_source(self):
        return conf.proto.Yql.UdfSource

    @cached_property
    def is_embedded(self):
        return conf.proto.YqlEmbedded.IsEmbedded

    @cached_property
    def mrjob_binary(self):
        return conf.proto.YqlEmbedded.MrjobBin

    @cached_property
    def udf_resolver_binary(self):
        return conf.proto.YqlEmbedded.UdfResolverBin

    @cached_property
    def udfs_dir(self):
        return conf.proto.YqlEmbedded.UdfsDir

    @cached_property
    def loglevel(self):
        return conf.proto.YqlEmbedded.LogLevel

    @cached_property
    def is_local(self):
        return False

    @cached_property
    def yql_server(self):
        return "yql.yandex.net"

    @cached_property
    def yql_port(self):
        return 443


class YQLRunnerTask(ConfMixin, JinjaResourceMixin, EmbeddedYqlMixin, ClientYqlMixin, TransactionalYtTask):

    """Yql BT-task with yql_runner as client/embedded"""

    _query = None

    @property
    def syntax(self):
        return 1

    @property
    def query_template(self):
        """Should be Jinja2 Query Template resource paths"""
        # TODO: May be allow raw template as string
        pass

    def render_query(self, **kwargs):
        """Create query from template"""
        if self._query is None:
            # cached call for make sure the VERBOSE query random dir, be equal to RUN query dirs
            self._query = "\n".join(
                (
                    super(YQLRunnerTask, self).render("header.sql.j2", **kwargs),
                    super(YQLRunnerTask, self).render(self.query_template, **kwargs),
                )
            )

        print(
            "\n-----START RENDERED QUERY-----\n"
            "{query}"
            "\n-----FINISH RENDERED QUERY-----\n".format(query=self._query),
            file=sys.stderr,
        )

        return self._query

    def get_context_data(self, **kwargs):
        """Fill context data for template rendering"""
        context = super(YQLRunnerTask, self).get_context_data(**kwargs)
        context.update(
            transaction_id=self.transaction_id,
            pool=self.pool or conf.yt.pool,
            yt_proxy=self.yt_proxy,
            salt=str(time.time()),
            query_title=sandbox_mark(self.get_title()),
            crypta_env=self.crypta_env,
            is_embedded=self.is_embedded,
            udf_source=self.udf_source,
            is_inside_test=("localhost" in self.yt_proxy),
        )
        return context

    def get_libs(self):
        """Return list of libraries with extra UD(A)Fs"""
        # TODO: no add all of exist lib, make only neaded
        return [
            {
                "name": "aggregation_lib.sql",
                "content": "/lib/aggregation_lib.sql",
                "disposition": "resource",
                "type": "library",
            },
            {
                "name": "metrica_lib.sql",
                "content": "/lib/metrica_lib.sql",
                "disposition": "resource",
                "type": "library",
            },
            {
                "name": "sn_id_parse_lib.sql",
                "content": "/lib/sn_id_parse_lib.sql",
                "disposition": "resource",
                "type": "library",
            },
            {"name": "config.sql", "content": "/lib/config.sql", "disposition": "resource", "type": "library"},
            {
                "name": "ut_utils_lib.sql",
                "content": "/lib/ut_utils_lib.sql",
                "disposition": "resource",
                "type": "library",
            },
        ]

    def get_title(self):
        """Retrun title to yql operation"""
        return "Crypta YQL [{env}] {text}".format(env=self.crypta_env, text=self.full_name())

    def run(self, **kwargs):
        query = self.render_query(**kwargs)
        if self.is_embedded:
            return self._run_embedded(query)
        else:
            return self._run_client(query)

    def validate(self, **kwargs):
        query = self.render_query(**kwargs)
        if self.is_embedded:
            return self._validate_embedded(query)
        else:
            return self._validate_client(query)
