import collections
import logging
import six

from crypta.lib.python.yt import yt_helpers

logger = logging.getLogger(__name__)

ReplicaStatus = collections.namedtuple("ReplicaStatus", ["id", "cluster", "path", "mode", "state", "replication_lag_time", "replicated_table_tracker_enabled"])


def format_replica_id(replica_id):
    return "#{}".format(replica_id)


def list_replicas(yt_client, replicated_table):
    replicas = yt_helpers.get_attribute(replicated_table, "replicas", yt_client)

    result = list()
    for replica_id, attrs in six.iteritems(replicas):
        result.append(ReplicaStatus(
            id=replica_id,
            cluster=attrs["cluster_name"],
            path=attrs["replica_path"],
            mode=attrs["mode"],
            state=attrs["state"],
            replication_lag_time=attrs["replication_lag_time"],
            replicated_table_tracker_enabled=attrs["replicated_table_tracker_enabled"],
        ))

    return result


def print_replicas(replicas):
    if replicas:
        _print_table(_format_replicas(replicas))


def _print_table(rows):
    if len(rows) == 0:
        return

    field_names = rows[0].keys()
    header = collections.OrderedDict((field, field) for field in field_names)
    table = [header] + rows

    column_widths = {field_name: max([len(str(row[field_name])) for row in table])
                     for field_name in field_names}
    for row in table:
        print(" ".join(str(row[field_name]).ljust(column_widths[field_name]) for field_name in field_names))


def _format_replicas(replicas):
    def format_field(field_name, field_value):
        return "#{}".format(field_value) if field_name == "id" else field_value

    return [collections.OrderedDict((k, format_field(k, v)) for k, v in six.iteritems(replica._asdict())) for replica in replicas]


def get_replica_id_by_path(master_client, master_table, replica_cluster, replica_table):
    replica = find_replica(master_client, master_table, replica_cluster, replica_table)
    assert replica, "There must be a replica object in {} for {} at {}".format(master_table, replica_table, replica_cluster)

    return replica.id


def find_replica(master_client, master_table, replica_cluster, replica_table):
    replicas = list_replicas(master_client, master_table)

    target_replicas = [r for r in replicas if (r.path == replica_table) and (r.cluster == replica_cluster)]
    assert len(target_replicas) <= 1, (
        "There must be one or zero replica objects in {} for {} at {} (found {})").format(master_table, replica_table, replica_cluster, len(target_replicas))
    if len(target_replicas) == 1:
        return target_replicas[0]
    else:
        return None


def _remove_replica_impl(master_client, master_table, replica_client, replica_table, remove_replica_table):
    replica_cluster = yt_helpers.get_cluster_name(replica_client)
    replica_id = get_replica_id_by_path(master_client, master_table, replica_cluster, replica_table)

    master_cluster = yt_helpers.get_cluster_name(master_client)

    replica_obj = "#{}".format(replica_id)

    logger.info("Removing replica object %s (%s at %s) from %s at %s", replica_obj, replica_table, replica_cluster, master_table, master_cluster)
    master_client.remove(replica_obj)

    success = True
    if remove_replica_table:
        if replica_client.exists(replica_table):
            if yt_helpers.get_attribute(replica_table, "type", replica_client) == "table":
                logger.info("Unmounting replica table %s at %s", replica_table, replica_cluster)
                replica_client.unmount_table(replica_table, sync=True)

                logger.info("Removing replica table %s at %s", replica_table, replica_cluster)
                replica_client.remove(replica_table)
            else:
                logger.error("%s at %s is not a table", replica_table, replica_cluster)
                success = False
        else:
            logger.error("Replica table %s does not exist at %s", replica_table, replica_cluster)
            success = False
    else:
        logger.info("Leaving replica table %s at %s intact", replica_table, replica_cluster)

    logger.info("Done" if success else "Done with errors")
    return success


def disconnect_replica(master_cluster, master_table, replica_cluster, replica_table):
    return _remove_replica_impl(master_cluster, master_table, replica_cluster, replica_table, remove_replica_table=False)


def remove_replica(master_cluster, master_table, replica_cluster, replica_table):
    return _remove_replica_impl(master_cluster, master_table, replica_cluster, replica_table, remove_replica_table=True)
