import concurrent.futures

from crypta.lib.python.yt.dyntables.kv_client import KvClient


class NotWaitingExecutor(concurrent.futures.ThreadPoolExecutor):
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.shutdown(wait=False)


class CrossDcKvClient(object):
    def __init__(self, main_kv_client, retry_kv_clients, retry_timeout):
        self.main_kv_client = main_kv_client
        self.retry_kv_clients = retry_kv_clients
        self.retry_timeout = retry_timeout

    def _do_with_retries(self, func):
        with NotWaitingExecutor(1 + len(self.retry_kv_clients)) as executor:
            main_future = executor.submit(func, self.main_kv_client)

            try:
                return main_future.result(self.retry_timeout.total_seconds())
            except Exception:
                futures = [main_future] + [executor.submit(func, kv_client) for kv_client in self.retry_kv_clients]

                for future in concurrent.futures.as_completed(futures, timeout=self.retry_timeout.total_seconds()):
                    if future.exception() is None:
                        return future.result()

        raise Exception("Requests to all DCs failed")

    def lookup_many(self, keys):
        return self._do_with_retries(lambda client: KvClient.lookup_many(client, keys))

    def lookup(self, key):
        return self._do_with_retries(lambda client: KvClient.lookup(client, key))

    def select(self, query):
        return self._do_with_retries(lambda client: KvClient.select(client, query))
