import logging

from crypta.lib.python.yt import yt_helpers

logger = logging.getLogger(__name__)


class KvMaster(object):
    class MasterLoggingAdapter(logging.LoggerAdapter):
        def process(self, msg, kwargs):
            return "[master {}:{}] {}".format(
                self.extra["cluster"], self.extra["path"],
                msg), kwargs

    def __init__(self, yt_client, path, schema, pivot_keys, bundle="default", enable_tablet_balancer=True, sync_replica_count=None, enable_replicated_table_tracker=False):
        self.yt_client = yt_client
        self.path = path
        self.schema = schema
        self.pivot_keys = pivot_keys
        self.bundle = bundle
        self.enable_tablet_balancer = enable_tablet_balancer
        self.sync_replica_count = sync_replica_count
        self.enable_replicated_table_tracker = enable_replicated_table_tracker
        self.logger = KvMaster.MasterLoggingAdapter(logger, {"cluster": self.yt_client.config['proxy']['url'], "path": self.path})

    @staticmethod
    def create(*args, **kwargs):
        master = KvMaster(*args, **kwargs)

        replicated_table_options = {"enable_replicated_table_tracker": master.enable_replicated_table_tracker}
        if master.sync_replica_count is not None:
            replicated_table_options["sync_replica_count"] = master.sync_replica_count

        attributes = {
            "tablet_cell_bundle": master.bundle,
            "schema": master.schema,
            "dynamic": True,
            "pivot_keys": master.pivot_keys,
            "enable_tablet_balancer": master.enable_tablet_balancer,
            "replicated_table_options": replicated_table_options,
        }
        master.logger.info("Creating replicated table")
        master.yt_client.create("replicated_table", master.path, attributes=attributes, recursive=True)
        master.logger.info("Mounting replicated table")
        master.yt_client.mount_table(master.path, sync=True)
        yt_helpers.wait_for_mounted(master.yt_client, master.path)
        return master

    @staticmethod
    def from_existing(yt_client, path):
        arguments_to_attrs = {
            "schema": "schema",
            "pivot_keys": "pivot_keys",
            "bundle": "tablet_cell_bundle",
            "enable_tablet_balancer": "enable_tablet_balancer"
        }
        attrs = yt_helpers.get_attributes(path, arguments_to_attrs.values(), yt_client)
        kwargs = {arg: attrs[name] for arg, name in arguments_to_attrs.items() if name in attrs}
        kwargs.setdefault("enable_tablet_balancer", True)

        master_logger = KvMaster.MasterLoggingAdapter(logger, {"cluster": yt_client.config['proxy']['url'], "path": path})
        master_logger.info("Initializing master from existing")

        return KvMaster(
            yt_client,
            path,
            **kwargs
        )
