import logging

from crypta.lib.python.yt import yt_helpers

logger = logging.getLogger(__name__)


class KvReplica(object):
    class ReplicaLoggingAdapter(logging.LoggerAdapter):
        def process(self, msg, kwargs):
            return "[master {}:{}] [replica {}:{}] {}".format(
                self.extra["master_cluster"], self.extra["master_path"],
                self.extra["replica_cluster"], self.extra["replica_path"],
                msg), kwargs

    def __init__(self, master, cluster_name, path, yt_client, sync=True, bundle="default", replica_id=None):
        self.master = master
        self.cluster_name = cluster_name
        self.path = path
        self.yt_client = yt_client
        self.sync = sync
        self.bundle = bundle
        self.replica_id = replica_id
        self.logger = KvReplica.ReplicaLoggingAdapter(logger, {
            "replica_cluster": self.yt_client.config['proxy']['url'],
            "replica_path": self.path,
            "master_cluster": self.master.yt_client.config['proxy']['url'],
            "master_path": self.master.path
        })

    @staticmethod
    def create(*args, **kwargs):
        replica = KvReplica(*args, **kwargs)
        replica.create_replica()
        replica.create_table()
        replica.prepare_dynamic()
        replica.enable_replica()
        return replica

    @staticmethod
    def from_existing(master, yt_client, path):
        replica_logger = KvReplica.ReplicaLoggingAdapter(logger, {
            "replica_cluster": yt_client.config['proxy']['url'],
            "replica_path": path,
            "master_cluster": master.yt_client.config['proxy']['url'],
            "master_path": master.path
        })
        replica_logger.info("Initializing replica from existing")

        replica_attrs = yt_helpers.get_attributes(path, ["upstream_replica_id", "tablet_cell_bundle"], yt_client)
        replica_id = replica_attrs["upstream_replica_id"]

        def get_master_attr(name):
            return master.yt_client.get("#{}/@{}".format(replica_id, name))

        return KvReplica(
            master,
            get_master_attr("cluster_name"),
            path,
            yt_client,
            sync=get_master_attr("mode") == "sync",
            bundle=replica_attrs["tablet_cell_bundle"],
            replica_id=replica_id
        )

    def _get_default_attrs(self):
        return dict(
            in_memory_mode="uncompressed",
            tablet_cell_bundle=self.bundle,
            enable_tablet_balancer=self.master.enable_tablet_balancer,
            enable_dynamic_store_read=True,
        )

    def create_table(self):
        self.logger.info("Creating empty replica table")
        self.yt_client.create(
            'table',
            self.path,
            attributes=dict(
                schema=self.master.schema,
                **self._get_default_attrs()
            ),
            recursive=True
        )

    def prepare_dynamic(self):
        self.logger.info("Switching replica table to dynamic and binding to upsream %s", self.replica_id)
        self.yt_client.alter_table(self.path, dynamic=True, upstream_replica_id=self.replica_id)

        self.logger.info("Resharding replica table")
        self.yt_client.reshard_table(self.path, pivot_keys=self.master.pivot_keys)

        self.logger.info("Mounting replica table")
        self.yt_client.mount_table(self.path, sync=True)
        yt_helpers.wait_for_mounted(self.yt_client, self.path)

    def create_replica(self):
        self.logger.info("Creating replica id")
        self.replica_id = self.master.yt_client.create(
            "table_replica",
            attributes=dict(
                table_path=self.master.path,
                cluster_name=self.cluster_name,
                replica_path=self.path
            )
        )

    def delete_replica(self):
        self.logger.info("Deleting replica")
        self.master.yt_client.remove("#{}".format(self.replica_id))

    def enable_replica(self):
        self.logger.info("Enabling replica")
        self.master.yt_client.alter_table_replica(self.replica_id, enabled=True, mode="sync" if self.sync else "async")

    def disable_replica(self):
        self.logger.info("Disabling replica")
        self.master.yt_client.alter_table_replica(self.replica_id, enabled=False)

    def replace_with_static(self, source_path):
        self.logger.info("Replacing replica state with %s", source_path)
        for attr, value in self._get_default_attrs().iteritems():
            self.yt_client.set("{}/@{}".format(source_path, attr), value)

        self.disable_replica()
        self.yt_client.unmount_table(self.path, sync=True)
        self.yt_client.move(source_path, self.path, force=True)
        self.prepare_dynamic()
        self.enable_replica()
