from __future__ import absolute_import
import copy
from yt import yson

from mapreduce.yt.python.table_schema import (
    extract_column_attributes,
    normalize_schema,
)
import six

import crypta.lib.python.arg_to_list as arg_to_list
from crypta.lib.python.yt.schema_utils.proto import get_schema_from_proto  # noqa

REQUIRED_FIELD = "required"
SORT_ORDER_FIELD = "sort_order"

VALID_YT_TYPES = {
    'boolean',
    'int8',
    'int16',
    'int32',
    'int64',
    'uint8',
    'uint16',
    'uint32',
    'uint64',
    'double',
    'string',
    'utf8',
    'any',
}


def is_column_sorted(column):
    return SORT_ORDER_FIELD in column


def get_sorted_columns(schema):
    return [x for x in schema if is_column_sorted(x)]


def get_unsorted_columns(schema):
    return {frozenset(six.iteritems(x)) for x in schema if not is_column_sorted(x)}


def are_schemas_equal(schema_1, schema_2):
    def set_defaults(schema):
        res = copy.copy(schema)

        for column in res:
            if REQUIRED_FIELD not in column:
                column[REQUIRED_FIELD] = False
        return res

    prepared_schema_1 = normalize_schema(extract_column_attributes(set_defaults(schema_1)))
    prepared_schema_2 = normalize_schema(extract_column_attributes(set_defaults(schema_2)))

    return prepared_schema_1.attributes == prepared_schema_2.attributes and \
           get_sorted_columns(prepared_schema_1) == get_sorted_columns(prepared_schema_2) and \
           get_unsorted_columns(prepared_schema_1) == get_unsorted_columns(prepared_schema_2)


def check_schemas_equality(schema_1, schema_2):
    if not are_schemas_equal(schema_1, schema_2):
        raise Exception("Schemas are not equal.\nschema_1: {}\nschema_2:{}".format(schema_1, schema_2))


def yt_schema_from_dict(schema_dict, sort_by=None):
    assert all(v in VALID_YT_TYPES for v in six.itervalues(schema_dict))
    yt_schema = []
    if sort_by:
        assert isinstance(sort_by, (six.string_types, list, tuple))
        sort_by = arg_to_list.arg_to_list(sort_by)

        for field_name in sort_by:
            yt_schema.append(
                {
                    'name': field_name,
                    'type': schema_dict[field_name],
                    'sort_order': 'ascending',
                }
            )

    for field_name, field_type in six.iteritems(schema_dict):
        if sort_by and field_name in sort_by:
            continue
        yt_schema.append({'name': field_name, 'type': field_type})

    return yt_schema


def yt_dyntable_schema_from_dict(schema_dict, sort_by, hash_expression=None, hash_name="Hash"):
    schema = yt_schema_from_dict(schema_dict, sort_by)
    if hash_expression:
        schema = [{
            'name': hash_name,
            'expression': hash_expression,
            'type': 'uint64',
            'sort_order': 'ascending',
        }] + schema

    schema = yson.YsonList(schema)
    schema.attributes["strict"] = True
    schema.attributes["unique_keys"] = True
    return schema


def get_strict_schema(fields):
    schema = yson.YsonList(fields)
    schema.attributes["strict"] = True
    return schema
