import argparse
import json
import logging
import os

import flask
from library.python.testing.recipe import declare_recipe, set_env
import yatest.common
from yatest.common import network

from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers

PID_FILE = "sandbox_api.pid"


def empty_answer():
    return flask.jsonify({
        "items": [],
        "total": 0,
        "limit": 1,
        "offset": 0
    })


def create_app(resources):
    app = flask.Flask(__name__)

    @app.route('/resource')
    def resource():
        expected_params = {"limit", "state", "order", "type", "attrs"}
        params = set(flask.request.args.keys())
        if params != expected_params:
            app.logger.info("Parameter list is different from expected: %s", params)
            return empty_answer()

        type = flask.request.args["type"]
        resource = resources.get(type)
        if resource is None:
            app.logger.info("Unknown type: %s", type)
            return empty_answer()

        release = json.loads(flask.request.args["attrs"]).get("released")
        if release not in resource:
            app.logger.info("Unknown release: %s", release)
            return empty_answer()

        return flask.jsonify({
            "items": [
                {
                    "attributes": {
                        "released": release
                    },
                    "id": resource[release],
                    "state": "READY",
                    "type": type
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        })

    return app


def parse_args(argv):
    parser = argparse.ArgumentParser(description='Sandbox API mock server')
    parser.add_argument('--config', help='Config file', required=True)
    return parser.parse_args(argv)


def run(host, port, argv):
    logging_helpers.configure_stdout_logger(logging.getLogger())
    args = parse_args(argv)
    resources = yaml_config.load(yatest.common.source_path(args.config))["resources"]

    app = create_app(resources)
    app.logger.info('--- START SERVER ---')
    app.run(host, port)


def start(argv):
    pm = network.PortManager()
    host = "localhost"
    port = pm.get_port()

    set_env("SANDBOX_API_URL_PREFIX", "http://{host}:{port}".format(host=host, port=port))

    pid = os.fork()
    if pid == 0:
        run(host, port, argv)
    else:
        with open(PID_FILE, "w") as f:
            f.write(str(pid))


def stop(_):
    with open(PID_FILE) as f:
        pid = int(f.read())
        os.kill(pid, 9)


def main():
    declare_recipe(start, stop)
