#include "calc_metrika_segments_job.h"

#include "calc_metrika_segments_reducer.h"
#include "parse_beh_profile_log_mapper.h"
#include "fields.h"

#include <crypta/lib/native/range/packs.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/processed_tables_tracker/processed_tables_tracker.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/tables_indexes.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lookalike/proto/beh_profile_regular_log_entry.pb.h>
#include <crypta/lookalike/proto/metrika_segment_binding.pb.h>
#include <crypta/lookalike/proto/error.pb.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/common.h>
#include <mapreduce/yt/util/temp_table.h>

#include <util/string/join.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;

namespace {
    void ReduceBindings(NYT::ITransactionPtr tx, const TString& freshTable, const TString& stateTable, size_t maxAudienceSize) {
        using namespace NCalcMetrikaSegments;
        tx->Sort(freshTable, freshTable, {NFields::SEGMENT_ID, NFields::TS});

        auto spec = NYT::TReduceOperationSpec().SortBy({NFields::SEGMENT_ID, NFields::TS}).ReduceBy({NFields::SEGMENT_ID});
        spec.AddInput<TMetrikaSegmentBinding>(freshTable);
        if (tx->Exists(stateTable)) {
            spec.AddInput<TMetrikaSegmentBinding>(stateTable);
        }
        spec.AddOutput<TMetrikaSegmentBinding>(NYT::TRichYPath(stateTable).Schema(NYT::CreateTableSchema<TMetrikaSegmentBinding>({NFields::SEGMENT_ID, NFields::TS})).OptimizeFor(NYT::OF_SCAN_ATTR));
        tx->Reduce(spec, new TCalcMetrikaSegmentsReducer(maxAudienceSize));
    }
}

int NCalcMetrikaSegments::CalcMetrikaSegmentsJob(TCalcMetrikaSegmentsJobConfig config, NLog::TLogPtr log) {
    const auto& ytProxy = config.GetYt().GetProxy();
    const auto& ytPool = config.GetYt().GetPool();
    const auto& metrikaSegmentsDstTablePath = config.GetMetrikaSegmentsDstTable();
    const auto& metrikaEcommerceDstTablePath = config.GetMetrikaEcommerceDstTable();
    const auto& mobileEventBindingsTablePath = config.GetMobileEventDstTable();

    auto client = NYT::CreateClient(ytProxy);
    NYT::TConfig::Get()->Pool = ytPool;

    TProcessedTablesTracker processedTablesTracker({.SourceDir = config.GetBehProfileRegularLogDir(), .StateTable = config.GetTrackTable()});
    const auto& sourceTables = processedTablesTracker.GetUnprocessedTables(client);
    log->info("List of source tables [\n{}\n]", JoinSeq(",\n", sourceTables));

    for (const auto& pack : GetPacks(sourceTables, config.GetTablePackSize())) {
        log->info("Parse source tables [\n{}\n]", JoinSeq(",\n", pack));

        const auto& errorsTablePath = TTimedYtPathGenerator(TShiftedClock::Now()).GetPath(config.GetErrorsDir());

        auto tx = client->StartTransaction();
        NYT::TTempTable freshMetrikaSegmentsBindingsTable(tx, "fresh_metrika_segments_bindings");
        NYT::TTempTable freshMetrikaEcommerceBindingsTable(tx, "fresh_metrika_ecommerce_bindings");
        NYT::TTempTable freshMobileEventBindingsTable(tx, "fresh_mobile_event_bindings");

        auto mapSpec = NYT::TMapOperationSpec();
        AddInputs<TBehProfileRegularLogEntry>(mapSpec, pack);

        TParseBehProfileLogMapper::TOutputIndexes::TBuilder outputBuilder;
        outputBuilder.AddOutput<TMetrikaSegmentBinding>(mapSpec, freshMetrikaSegmentsBindingsTable.Name(), TParseBehProfileLogMapper::EOutputTables::MetrikaSegmentBindings);
        outputBuilder.AddOutput<TMetrikaSegmentBinding>(mapSpec, freshMetrikaEcommerceBindingsTable.Name(), TParseBehProfileLogMapper::EOutputTables::MetrikaEcommerceBindings);
        outputBuilder.AddOutput<TMetrikaSegmentBinding>(mapSpec, freshMobileEventBindingsTable.Name(), TParseBehProfileLogMapper::EOutputTables::MobileEventBindings);
        outputBuilder.AddOutput<TError>(mapSpec, errorsTablePath, TParseBehProfileLogMapper::EOutputTables::Errors);

        tx->Map(mapSpec, new TParseBehProfileLogMapper(outputBuilder.GetIndexes()));

        ReduceBindings(tx, freshMetrikaSegmentsBindingsTable.Name(), metrikaSegmentsDstTablePath, config.GetMaxAudienceSize());
        ReduceBindings(tx, freshMetrikaEcommerceBindingsTable.Name(), metrikaEcommerceDstTablePath, config.GetMaxAudienceSize());
        ReduceBindings(tx, freshMobileEventBindingsTable.Name(), mobileEventBindingsTablePath, config.GetMaxAudienceSize());

        processedTablesTracker.AddProcessedTables(tx, pack);

        tx->Commit();

        SetTtl(client, errorsTablePath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);
    }

    return 0;
}
