import functools
import logging

import nirvana.mr_job_context as nv

from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import yt_helpers
from crypta.lookalike.lib.python.utils.config import config
from crypta.lookalike.services.custom_lookalike.lib import (
    lookalike,
    sample,
)


logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Custom lookalike')

    context = nv.context()
    parameters = context.parameters
    inputs = context.get_mr_inputs()

    sample_table = inputs.get('sample_table').get_path()
    group_features_table = inputs.get('group_features_table').get_path() if inputs.has('group_features_table') else None
    lookalike_table = inputs.get('lookalike_table').get_path() if inputs.has('lookalike_table') else None
    sizes_table = inputs.get('sizes_table').get_path() if inputs.has('sizes_table') else None
    model_dir = inputs.get('model_dir').get_path() if inputs.has('model_dir') else None

    yt_client = yt_helpers.get_yt_client_from_nv_parameters(
        nv_parameters=parameters,
        tmp_directory=config.COMMON_TMP_DIRECTORY,
    )
    yql_client = yql_helpers.get_yql_client_from_nv_parameters(parameters, config.COMMON_YQL_TMP_DIRECTORY)

    tasks_dict = {
        'lookalike_by_sample': functools.partial(
            lookalike.get_by_sample,
            yt_client=yt_client,
            yql_client=yql_client,
            sample_table=sample_table,
            lookalike_table=lookalike_table,
            sizes_table=sizes_table,
            model_dir=model_dir,
        ),
        'get_group_features': functools.partial(
            sample.get_group_features,
            yt_client=yt_client,
            yql_client=yql_client,
            sample_table=sample_table,
            group_features_table=group_features_table,
            model_dir=model_dir,
        ),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
