#include <crypta/lib/native/event_processing/worker.h>
#include <crypta/lib/native/grpc/async_client/proto/async_client_config.pb.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>
#include <crypta/lib/native/proto_secrets/remove_secrets.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/stats/utils.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/tvm/create_tvm_client.h>
#include <crypta/lib/native/yaml/config/config.h>
#include <crypta/lib/native/yaml/ytree_node/utils.h>
#include <crypta/lib/native/yt/dyntables/async_database/kv_async_database.h>
#include <crypta/lookalike/services/lal_manager/lib/config/config.pb.h>
#include <crypta/lookalike/services/lal_manager/lib/processor_factory.h>
#include <crypta/lookalike/services/lal_manager/lib/processors/lal_processor.h>
#include <crypta/siberia/bin/common/siberia_client/cpp/siberia_client.h>
#include <crypta/siberia/bin/custom_audience/fast/bin/clients/cpp/ca_async_client.h>

#include <mapreduce/interface/client.h>
#include <mapreduce/yt/common/config.h>
#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NLog;
using namespace NCrypta::NLookalike::NLalManager;

int main(int argc, const char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    try {
        TShiftedClock::FreezeTimestampFromEnv();
        const auto& config = ParseYamlConfig<TConfig>(argc, argv);

        RegisterLogs(config.GetLogs());

        auto log = GetLog("main");
        log->info("Config:\n{}", NProtoSecrets::GetCopyWithoutSecrets(config).DebugString());

        auto statsRegistry = Singleton<TStatsRegistry>();
        TStatsWriter statsWriter(*statsRegistry, TDuration::Minutes(1));
        if (config.HasGraphite()) {
            statsWriter.AddSink(MakeHolder<TGraphiteStatsSink>(config.GetGraphite()));
        }
        if (config.HasSolomon()) {
            statsWriter.AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
        }
        statsWriter.AddSink(MakeHolder<TLogStatsSink>("graphite"));
        statsWriter.Start();

        const auto& statsSettings = GetStatsSettings(config.GetStats());

        TStats pqlibStats("pqlib", statsSettings);
        const auto& logbrokerConfig = config.GetLogbroker();
        auto pqLibLogger = MakeIntrusive<NCrypta::NLog::TPqSpdlogWrapper>(logbrokerConfig.GetPqlibLogName());
        NPQ::TConsumer consumer(
            NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
            pqLibLogger,
            logbrokerConfig.GetConsumer(),
            NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), pqLibLogger),
            pqlibStats
        );

        const auto& tvmClient = CreateTvmClient(config.GetTvm());

        const auto& siberiaClientConfig = config.GetSiberiaClient();
        NSiberia::TSiberiaClient siberiaClient(siberiaClientConfig.GetHost(), siberiaClientConfig.GetPort(), TDuration::Seconds(siberiaClientConfig.GetTimeoutSec()), tvmClient, MakeRetryOptions(siberiaClientConfig.GetRetryOptions()));

        NSiberia::NCustomAudience::TCaAsyncClient caClient(config.GetCustomAudienceClient());

        auto ytClient = NYT::CreateClient(config.GetStaticYtClientProxy());

        const auto& lalDatabaseConfig = config.GetLalDatabase();

        TStats lalDatabaseStats("lal_database", statsSettings);
        NYtDynTables::TKvDatabase lalDatabase(
            MakeAtomicShared<NYtDynTables::TKvAsyncDatabase>(lalDatabaseConfig.GetClient(), lalDatabaseConfig.GetTable(), lalDatabaseConfig.GetModifyRows()),
            lalDatabaseStats
        );

        TStats workerStats("worker", statsSettings);
        TLalProcessor::TConfig lalProcessorConfig {
            .YtClient = ytClient,
            .GoalAudienceTablePath = config.GetGoalAudienceTablePath(),
            .MobileEventTablePath = config.GetMobileEventTablePath(),
            .MetrikaSegmentsTablePath = config.GetMetrikaSegmentsTablePath(),
            .MetrikaEcommerceTablePath = config.GetMetrikaEcommerceTablePath(),
            .MetrikaCounterAudienceTablePath = config.GetMetrikaCounterAudienceTablePath(),
            .AudienceSegmentsTablePath = config.GetAudienceSegmentsTablePath(),
            .CdpSegmentsTablePath = config.GetCdpSegmentsTablePath(),
            .CustomAudiencesTablePath = config.GetCustomAudiencesTablePath(),
            .LalDatabase = lalDatabase,
            .SiberiaClient = siberiaClient,
            .CaClient = caClient,
            .MaxIdsToDescribe = config.GetMaxIdsToDescribe(),
            .DescribingMode = config.GetDescribingMode(),
            .RedescribeTtl = TDuration::Days(config.GetRedescribeTtlDays()),
            .StatsCheckInterval = TDuration::Seconds(config.GetStatsCheckIntervalSeconds())
        };
        TProcessorFactory processorFactory(std::move(lalProcessorConfig), workerStats);

        NPQ::TCookieQueue cookiesToCommit;
        auto worker = NYT::New<NEventProcessing::TWorker>(processorFactory, cookiesToCommit, workerStats);
        auto threadPool = NYT::New<NYT::NConcurrency::TThreadPool>(config.GetWorker().GetThreads(), "worker");

        NPQ::TReader reader(
            TDuration::MilliSeconds(config.GetReader().GetReadTimeoutMs()),
            consumer,
            cookiesToCommit,
            [&worker, &threadPool] (auto&& readResult) {
                threadPool->GetInvoker()->Invoke(BIND([&worker, readResult = std::move(readResult)] () mutable {
                    worker->Process(std::move(readResult));
                }));
            },
            NLog::GetLog(logbrokerConfig.GetPqlibLogName())
        );
        reader.SetOnFinishHandler([&stopSignalWaiter] () {
            stopSignalWaiter.Signal();
        });

        log->info("Start");
        reader.Start();

        stopSignalWaiter.Wait();

        log->info("Stop");
    } catch (const yexception& e) {
        LogException(e.what());
        return 1;
    } catch (const std::exception& e) {
        LogException(e.what());
        return 1;
    } catch (...) {
        NLog::LogException("Unknown exception");
        return 1;
    }

    Cerr << "Shutting down YT library" << Endl;
    NYT::Shutdown();

    Cerr << "Exiting" << Endl;
    return 0;
}
