#pragma once

#include <crypta/lib/native/event_processing/processor.h>
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/yt/dyntables/async_database/kv_database.h>
#include <crypta/lib/proto/identifiers/id.pb.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/lookalike/proto/lal_parent.pb.h>
#include <crypta/lookalike/proto/lal_state.pb.h>
#include <crypta/siberia/bin/common/siberia_client/cpp/siberia_client.h>
#include <crypta/siberia/bin/custom_audience/fast/bin/clients/cpp/ca_async_client.h>

#include <library/cpp/tvmauth/client/facade.h>
#include <mapreduce/yt/interface/client.h>

#include <util/string/builder.h>

namespace NCrypta::NLookalike::NLalManager {
    class TStaticYtException : public yexception {
    };

    class TAudienceSegmentsYtException : public yexception {
    };

    class TLalProcessor : public NEventProcessing::IProcessor {
    public:
        struct TConfig {
            NYT::IClientPtr YtClient;
            const TString GoalAudienceTablePath;
            const TString MobileEventTablePath;
            const TString MetrikaSegmentsTablePath;
            const TString MetrikaEcommerceTablePath;
            const TString MetrikaCounterAudienceTablePath;
            const TString AudienceSegmentsTablePath;
            const TString CdpSegmentsTablePath;
            const TString CustomAudiencesTablePath;
            NYtDynTables::TKvDatabase& LalDatabase;
            const NSiberia::TSiberiaClient& SiberiaClient;
            NSiberia::NCustomAudience::TCaAsyncClient& CaClient;
            const size_t MaxIdsToDescribe = 0;
            const TString DescribingMode;

            TDuration RedescribeTtl;
            TDuration StatsCheckInterval;
        };

        struct TDescriptionWithMeta {
            TDescription Description;
            size_t Size = 0;
        };

        TLalProcessor(TConfig config, TString name, ::TStats& stats);

    protected:
        struct TParentAudience {
            TIds IdsToDescribe;
            size_t Size = 0;
        };

        virtual bool Process(TMaybe<TLalState> currentLalState) = 0;
        virtual ui64 GetLalId() const = 0;

        TMaybe<TDescriptionWithMeta> DescribeParent(const TLalParent& parent);
        void WriteLalState(const TLalState& lalState);
        void RemoveLalState();
        bool NeedToDescribe(const TLalState& state) const;

        template <typename... Args>
        void LogMessage(spdlog::level::level_enum level, const TString& fmt, const Args&... args) {
            const TString message = TStringBuilder() << "[" << Name <<  "][Lal id = " << GetLalId() << "] " << fmt;
            Log->log(level, message.c_str(), args...);
        }

        template <typename... Args>
        void LogInfo(const TString& fmt, const Args&... args) {
            LogMessage(spdlog::level::level_enum::info, fmt, args...);
        }

        template <typename... Args>
        void LogError(const TString& fmt, const Args&... args) {
            LogMessage(spdlog::level::level_enum::err, fmt, args...);
        }

        TConfig Config;
        const ui64 RedescribeTsThreshold = 0;
        const TString Name;
        ::TStats& Stats;

    private:
        bool Process() override;
        TMaybe<TLalState> GetLalState();
        NLab::TUserDataStats DescribeIds(const TIds& ids);
        TMaybe<TDescriptionWithMeta> DescribeParentWithAudience(const TLalParent& parent);
        TMaybe<TDescriptionWithMeta> DescribeСustomAudience(const TLalParent& parent);
        NLab::TUserDataStats GetUserSetStats(ui64 userSetId);
        TMaybe<TDescriptionWithMeta> GetAudienceSegmentStats(ui64 parentId);
        TParentAudience GetParentAudience(const TLalParent& parent);
        TMaybe<TDescriptionWithMeta> DescribeParentAudience(const TLalParent::EType& parentType, const TParentAudience& parentAudience);

        template<typename TIdExtractor>
        TParentAudience GetAudience(ui64 id, const TString& table) {
            TParentAudience parentAudience;

            try {
                auto tx = Config.YtClient->StartTransaction();
                if (tx->Exists(table)) {
                    const auto richPath = NYT::TRichYPath(table).AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(id)));
                    auto reader = tx->CreateTableReader<typename TIdExtractor::TBinding>(richPath);
                    for (; reader->IsValid(); reader->Next()) {
                        const auto& binding = reader->GetRow();
                        auto extractedId = TIdExtractor::GetId(binding);

                        if (!NIdentifiers::TGenericID(extractedId.GetType(), extractedId.GetValue()).IsSignificant()) {
                            continue;
                        }

                        if (parentAudience.Size < Config.MaxIdsToDescribe) {
                            parentAudience.IdsToDescribe.AddIds()->Swap(&extractedId);
                        }

                        ++parentAudience.Size;
                    }
                }
                tx->Commit();
            } catch (const yexception& e) {
                ythrow TStaticYtException() << "Bindings table error: " << e.what();
            }

            return parentAudience;
        }

        NLog::TLogPtr Log;
    };
}
