import os
import traceback

from library.python.protobuf.json import proto2json
from yt.wrapper import aggregator

from crypta.lib.python import time_utils
from crypta.lib.python.lb_pusher import logbroker
from crypta.lookalike.proto import lal_state_pb2
from crypta.lookalike.services.lal_manager.commands import (
    describe_lal_cmd_pb2,
    lal_cmd_pb2,
)


@aggregator
class RefreshLalsMapper(object):
    def __init__(self, config, tvm_id):
        self.config = config
        self.update_ts_threshold = time_utils.get_current_time() - config.UpdateIntervalSec
        self.tvm_id = tvm_id

    @staticmethod
    def get_serialized_describe_command(lal_id):
        cmd = lal_cmd_pb2.TLalCmd(DescribeLalCmd=describe_lal_cmd_pb2.TDescribeLalCmd(LalId=long(lal_id)))
        return proto2json.proto2json(cmd)

    def __call__(self, rows):
        with logbroker.PQClient(
                self.config.Logbroker.Url,
                self.config.Logbroker.Port,
                tvm_id=self.tvm_id,
                tvm_secret=os.environ.get('YT_SECURE_VAULT_TVM_SECRET')
        ) as client:
            with logbroker.BatchingPQWriter(client.get_writer(self.config.FastTopic)) as fast_topic_writer, \
                    logbroker.BatchingPQWriter(client.get_writer(self.config.SlowTopic)) as slow_topic_writer:
                for row in rows:
                    try:
                        lal_state = lal_state_pb2.TLalState()
                        lal_state.ParseFromString(row["value"])
                        lal_id = row["key"]

                        if not lal_state.HasField("Description"):
                            fast_topic_writer.write(self.get_serialized_describe_command(lal_id))
                        elif lal_state.Description.Ts <= self.update_ts_threshold:
                            slow_topic_writer.write(self.get_serialized_describe_command(lal_id))
                    except Exception:
                        yield dict(row, error=traceback.format_exc())
