#include "direct_entry_converter.h"

#include <crypta/lib/native/retargeting_ids/retargeting_id.h>
#include <crypta/lookalike/lib/native/goal_type_converter.h>

#include <util/string/builder.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NLalSynchronizer;

namespace {
    static const TString RETARGETING_ID_TYPE = "retargeting_id";
    static const TString CUSTOM_AUDIENCE_TYPE = "custom_audience";

    TInputLalEntry GetInputLalEntry(ui64 lalId, TLalParent::EType parentType, ui64 parentId, TMaybe<TString> parentRule = Nothing()) {
        TInputLalEntry result;

        result.SetLalId(lalId);
        result.SetParentType(parentType);
        result.SetParentId(parentId);
        if (parentRule) {
            result.SetParentRule(parentRule.GetRef());
        }
        return result;
    }
}

TMaybe<TInputLalEntry> TDirectEntryConverter::GetLalEntry(const TEntry& entry) {
    const auto& lalId = entry.GetLalId();

    auto getErrorMessage = [lalId](const TString& message) {
        return TStringBuilder() << "[Lal id = " << lalId << "] " << message;
    };

    if (entry.GetIsActive()) {
        Y_ENSURE(NRetargetingIds::TRetargetingId(lalId).GetType() == NRetargetingIds::EType::DirectLalSegment, getErrorMessage("Invalid lal id"));

        const auto& parentType = entry.GetParentType();
        const auto& parentId = entry.GetGoalId();

        if (!entry.HasParentType() || parentType == RETARGETING_ID_TYPE) {
            const auto& retargetingId = NRetargetingIds::TRetargetingId(parentId);
            const auto& lalParentType = NGoalTypeConverter::ConvertRetargetingIdTypeToLalParentType(retargetingId.GetType());
            Y_ENSURE(lalParentType != TLalParent::UNSUPPORTED, getErrorMessage(TStringBuilder() << "Unsupported parent type: " << retargetingId.GetType()));
            return GetInputLalEntry(lalId, lalParentType, retargetingId.GetEntityId());
        } else if (parentType == CUSTOM_AUDIENCE_TYPE) {
            const auto& parentRule = entry.GetParentRule();
            Y_ENSURE(!parentRule.empty(), getErrorMessage("Parent rule is empty"));
            return GetInputLalEntry(lalId, TLalParent::CUSTOM_AUDIENCE, parentId, entry.GetParentRule());
        } else {
            ythrow yexception() << getErrorMessage(TStringBuilder() << "Unknown parent type: " << parentType);
        }
    }

    return Nothing();
}
